#!/bin/sh
#
# scw Description = Check and remount remote mount points
# scw Schedule = * * * * *
# scw MaxRunTime = 120
# scw SuccessInterval = 180
# scw SilentConcurrency = true
#
# For any NFS or CIFS filesystems that are either already mounted or are
# supposed to be automatically mounted on boot, check that they are still
# mounted (in case of stale NFS handle and so on).  Any which are not are
# unmounted and remounted.  Exits with failure if any remain unmounted.
#
# This script won't cope with mount points with whitespace in their paths.
#
# If monitored through the item list file, this will raise an alert if a
# mount point remains broken after 3 minutes.  An operator can then review
# this item's logs to see what failed and when.
#

timeoutCommand="timeout 10"
exitStatus=0

for mountPoint in $(awk '$3~/^(nfs|cifs)/ && !/^[[:space:]]*#/ {print $2}' /etc/fstab); do
	# Skip if it's not mounted on boot and isn't currently mounted.
	if awk -v m="${mountPoint}" '$2==m {print $4}' /etc/fstab | grep -Fq 'noauto'; then
		awk -v m="${mountPoint}" '$2==m {print}' /proc/mounts | grep -Eq . || continue
	fi

	printf "%s %s\n" "notice" "${mountPoint}: checking" >&3

	isMounted=false
	${timeoutCommand} mountpoint -q "${mountPoint}" && isMounted=true

	if ! ${isMounted}; then
		# If not mounted - warn, and attempt to remount.
		printf "%s %s\n" "warning" "${mountPoint}: not mounted" >&3
		printf "%s %s\n" "notice" "${mountPoint}: attempting remount" >&3
		${timeoutCommand} umount "${mountPoint}" \
		|| ${timeoutCommand} umount -l "${mountPoint}" \
		|| ${timeoutCommand} mount "${mountPoint}"
		${timeoutCommand} mountpoint -q "${mountPoint}" && isMounted=true
	fi
	if ! ${isMounted}; then
		# If still not mounted - report the error.
		printf "%s %s\n" "error" "${mountPoint}: not mounted, remediation failed" >&3
		exitStatus=1
	else
		# Report that the mount is OK.
		printf "%s %s\n" "ok" "${mountPoint}: is mounted" >&3
	fi
done

exit ${exitStatus}
