/*
 * Functions for capturing and recording the output of a command.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <sys/file.h>
#include <errno.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/un.h>


/*
 * Record the output of the running command by reading the command's stdout,
 * stderr, and status from pipes.  If the MaxRunTime is reached, we
 * terminate the process.
 *
 * Returns zero on success, or an SCW_EXIT_* value.  The file descriptors
 * passed in are closed before returning.
 */
int captureCommandOutputViaPipe(struct scwState *state, struct scwCommandProcess *process)
{
	bool commandRunning, allDescriptorsEnded, sentTerm;
	size_t descriptorIndex;
	struct {
		char buffer[SCW_MAX_READBUFFER];	/* flawfinder: ignore */
		size_t position;
		int fd;
		scwOutputStreamId whichStream;
		bool eof;
	} commandOutput[3];
	int retcode = 0;
	double elapsed;

	/*
	 * flawfinder warns about buffer[]; the operations which use it are
	 * bounded to SCW_MAX_READBUFFER, so it does not overrun, and we do
	 * not assume null-termination.
	 */

	/*
	 * Read the output of the command, until all of its streams close
	 * and it exits.  If the MaxRunTime is reached, we send it a
	 * SIGTERM, then a SIGKILL.
	 */
	commandRunning = true;
	allDescriptorsEnded = false;
	sentTerm = false;

	memset(commandOutput, 0, sizeof(commandOutput));
	commandOutput[0].fd = process->stdoutDescriptor;
	commandOutput[0].whichStream = SCW_STREAM_STDOUT;
	commandOutput[0].eof = process->stdoutDescriptor < 0 ? true : false;
	commandOutput[1].fd = process->stderrDescriptor;
	commandOutput[1].whichStream = SCW_STREAM_STDERR;
	commandOutput[1].eof = process->stderrDescriptor < 0 ? true : false;
	commandOutput[2].fd = process->statusDescriptor;
	commandOutput[2].whichStream = SCW_STREAM_STATUS;
	commandOutput[2].eof = process->statusDescriptor < 0 ? true : false;

	while (commandRunning || !allDescriptorsEnded) {
		fd_set readSet;
		struct timeval selectTimeout;
		int highestDescriptor;
		int waited, status;

		/*
		 * Read from the command's outputs.
		 */

#if SPLINT
		/* splint doesn't like FD_ZERO. */
		memset(&readSet, 0, sizeof(readSet));
#else				/* !SPLINT */
		FD_ZERO(&readSet);
#endif				/* !SPLINT */
		allDescriptorsEnded = true;
		highestDescriptor = 0;
		for (descriptorIndex = 0; descriptorIndex < 3; descriptorIndex++) {
			if (commandOutput[descriptorIndex].eof)
				continue;
			allDescriptorsEnded = false;

			/*@-shiftnegative@ */
			FD_SET(commandOutput[descriptorIndex].fd, &readSet);
			/*@+shiftnegative@ */

			if (commandOutput[descriptorIndex].fd > highestDescriptor)
				highestDescriptor = commandOutput[descriptorIndex].fd;
		}

		selectTimeout.tv_sec = 0;
		selectTimeout.tv_usec = 100000;
		/*@-nullpass@ *//* splint disagrees about NULL in select(). */
		if (select(1 + highestDescriptor, &readSet, NULL, NULL, &selectTimeout) < 0) {
			perror(PACKAGE_NAME);
#if SPLINT
			/* splint doesn't like FD_ZERO. */
			memset(&readSet, 0, sizeof(readSet));
#else				/* !SPLINT */
			FD_ZERO(&readSet);
#endif				/* !SPLINT */
		}
		/*@+nullpass@ */

		/*@-mustfreefresh@ */
		/* splint says "newline" leaks memory here, but it doesn't. */
		for (descriptorIndex = 0; descriptorIndex < 3; descriptorIndex++) {
			size_t maxBytesToRead;
			ssize_t bytesRead;
			char *newline;

			if (commandOutput[descriptorIndex].eof)
				continue;
			/*@-shiftnegative@ */
			/*@-type@ */
			if (0 == FD_ISSET(commandOutput[descriptorIndex].fd, &readSet))
				continue;
			/*@+type@ */
			/*@+shiftnegative@ */

			maxBytesToRead = SCW_MAX_READBUFFER - commandOutput[descriptorIndex].position;
			if (0 == maxBytesToRead) {
				/* Full buffer - output it now. */
				recordOutput(state, process->settings, process->pid,
					     commandOutput[descriptorIndex].whichStream,
					     commandOutput[descriptorIndex].buffer,
					     commandOutput[descriptorIndex].position);
				commandOutput[descriptorIndex].position = 0;
				maxBytesToRead = SCW_MAX_READBUFFER;
			}

			bytesRead = read(commandOutput[descriptorIndex].fd,	/* flawfinder: ignore */
					 commandOutput[descriptorIndex].buffer +
					 commandOutput[descriptorIndex].position, maxBytesToRead);
			/* flawfinder - read() is bounded, to stay within the buffer. */
			if (bytesRead < 0 && errno == EINTR) {
				perror(PACKAGE_NAME);
				continue;
			} else if (bytesRead < 0 && errno != EINTR) {
				perror(PACKAGE_NAME);
				commandOutput[descriptorIndex].eof = true;
				if (commandOutput[descriptorIndex].position > 0) {
					debug("%s", "error and EOF - emitting line so far");
				}
			} else if (0 == bytesRead) {
				debug("%s: %d", "EOF on fd", descriptorIndex);
				commandOutput[descriptorIndex].eof = true;
				if (commandOutput[descriptorIndex].position > 0) {
					debug("%s", "EOF - emitting line so far");
				}
			} else {
				commandOutput[descriptorIndex].position += bytesRead;
			}

			/* Output line by line. */
			newline =
			    memchr(commandOutput[descriptorIndex].buffer, (int) '\n',
				   commandOutput[descriptorIndex].position);
			while (NULL != newline) {
				size_t newlineOffset = newline - commandOutput[descriptorIndex].buffer;

				recordOutput(state, process->settings, process->pid,
					     commandOutput[descriptorIndex].whichStream,
					     commandOutput[descriptorIndex].buffer, newlineOffset);

				newlineOffset++;
				if (newlineOffset < commandOutput[descriptorIndex].position) {
					(void) memmove(commandOutput[descriptorIndex].buffer,
						       &(commandOutput[descriptorIndex].buffer[newlineOffset]),
						       commandOutput[descriptorIndex].position - newlineOffset);
					commandOutput[descriptorIndex].position -= newlineOffset;
				} else {
					commandOutput[descriptorIndex].position = 0;
				}

				newline = NULL;
				if (commandOutput[descriptorIndex].position > 0) {
					newline =
					    memchr(commandOutput[descriptorIndex].buffer, (int) '\n',
						   commandOutput[descriptorIndex].position);
				}
			}
		}
		/*@+mustfreefresh@ */


		/*
		 * (D4) Terminate the command if MaxRunTime is exceeded.
		 */
		if (commandRunning && process->settings->numMaxRunTime > 0) {
			elapsed = difftime(time(NULL), process->startTime);
			if (elapsed >= (double) (1 + process->settings->numMaxRunTime)) {
				/* 1 second past MaxRunTime, send SIGKILL. */
				debug("%s", "sending SIGKILL");
				(void) kill(process->pid, SIGKILL);
				commandRunning = false;
				process->timedOut = true;
				retcode = SCW_EXIT_RUN_TIMEOUT;
			} else if ((!sentTerm) && elapsed >= (double) (process->settings->numMaxRunTime)) {
				/* First send one SIGTERM. */
				debug("%s", "sending SIGTERM");
				(void) kill(process->pid, SIGTERM);
				sentTerm = true;
				process->timedOut = true;
				retcode = SCW_EXIT_RUN_TIMEOUT;
			}
		}

		/*
		 * Transmit any delayed spools (output lines to transmit to
		 * a URL after a short delay - HTTPInterval).
		 */
		transmitDelayedUrlSpools(state, process->settings);

		/*
		 * Check whether the command process has exited.
		 */

		if (commandRunning) {
			status = 0;
			/*@-type@ */
			/* splint disagreement about __pid_t vs pid_t. */
			waited = waitpid(process->pid, &status, WNOHANG);
			/*@+type@ */
			if (waited < 0 && errno == EINTR) {
				perror(PACKAGE_NAME);
				continue;
			} else if (waited < 0 && errno != EINTR) {
				perror(PACKAGE_NAME);
				process->exitStatus = 1;
				break;
			}

			/*@-predboolint@ */
			/*@-boolops@ */
			/* splint sees WIF...() as non-boolean. */
			if (waited > 0 && WIFEXITED(status)) {
				process->exitStatus = WEXITSTATUS(status);
				commandRunning = false;
				debug("%s: %d", "command exited", process->exitStatus);
			} else if (waited > 0 && WIFSIGNALED(status)) {
				process->exitStatus = 126;
				commandRunning = false;
				debug("%s", "command terminated by signal");
			}
			/*@+predboolint@ *//*@+boolops@ */
		}
	}

	/* Record any incomplete lines still buffered. */
	for (descriptorIndex = 0; descriptorIndex < 3; descriptorIndex++) {
		if (0 == commandOutput[descriptorIndex].position)
			continue;
		recordOutput(state, process->settings, process->pid, commandOutput[descriptorIndex].whichStream,
			     commandOutput[descriptorIndex].buffer, commandOutput[descriptorIndex].position);
	}

	/* Close the command's output file descriptors. */
	if (process->stdoutDescriptor >= 0)
		(void) close(process->stdoutDescriptor);
	if (process->stderrDescriptor >= 0)
		(void) close(process->stderrDescriptor);
	if (process->statusDescriptor >= 0)
		(void) close(process->statusDescriptor);

	return retcode;
}


/*
 * Handle alarm signals by doing nothing.
 *
 * Note that we have to use a signal handler like this, instead of using
 * SIG_IGN, because if we ignore the signal entirely, it does nothing,
 * including not interrupting blocking recvfrom() calls - which is what
 * we're using alarm signals for in the first place.
 */
static void ignoreAlarm( /*@unused@ */  __attribute__((unused))
			int s)
{
	debug("%s", "SIGALRM received");
	/* Do nothing. */
}


/*
 * Record the output of the running command, by receiving messages from a
 * Unix socket each of its output streams is writing to.  If the MaxRunTime
 * is reached, we terminate the process.
 *
 * Returns zero on success, or an SCW_EXIT_* value.  The file descriptors
 * passed in are closed before returning.
 */
int captureCommandOutputViaUnixSocket(struct scwState *state, struct scwCommandProcess *process)
{
	char receiveBuffer[SCW_MAX_READBUFFER];	/* flawfinder: ignore */
	size_t descriptorIndex;
	bool commandRunning, sentTerm;
	struct {
		char buffer[SCW_MAX_READBUFFER];	/* flawfinder: ignore */
		size_t position;
		scwOutputStreamId whichStream;
	} commandOutput[3];
	int retcode = 0;
	double elapsed;
	time_t now, whenOutputLastSeen;

	/*
	 * flawfinder warns about receiveBuffer[] and buffer[]; the
	 * operations which use these are bounded to SCW_MAX_READBUFFER, and
	 * we do not assume null-termination, so neither will overrun.
	 */

	/*
	 * Read the output of the command, until all of its streams close
	 * and it exits.  If the MaxRunTime is reached, we send it a
	 * SIGTERM, then a SIGKILL.
	 */
	commandRunning = true;
	sentTerm = false;

	/*
	 * Handle SIGALRM by doing nothing, so we can use alarms to
	 * interrupt blocking writes (returning EINTR).
	 */
	/*@-compdestroy@ */
	/* splint warns about leaking sa_mask, which we can't do anything about. */
	{
		struct sigaction signalAction;
		memset(&signalAction, 0, sizeof(signalAction));
		signalAction.sa_handler = ignoreAlarm;
		(void) sigemptyset(&(signalAction.sa_mask));
		signalAction.sa_flags = 0;
		(void) sigaction(SIGALRM, &signalAction, NULL);
	}
	/*+compdestroy@ */

	memset(receiveBuffer, 0, sizeof(receiveBuffer));

	memset(commandOutput, 0, sizeof(commandOutput));
	commandOutput[0].whichStream = SCW_STREAM_STDOUT;
	commandOutput[1].whichStream = SCW_STREAM_STDERR;
	commandOutput[2].whichStream = SCW_STREAM_STATUS;

	now = time(NULL);
	whenOutputLastSeen = now;

	while (process->combinedDescriptor >= 0 && (commandRunning || whenOutputLastSeen > (now - 1))) {
		struct sockaddr_un peerAddress;
		socklen_t addressLength;
		ssize_t receivedBytes;
		size_t senderIndex = 99;
		size_t readOffset, bytesRemaining;
		int waited, status;

		/*
		 * Read from the command's outputs.
		 */

		memset(&peerAddress, 0, sizeof(peerAddress));
		senderIndex = 99;

		(void) alarm(1);	    /* Don't wait forever. */
		addressLength = (socklen_t) sizeof(peerAddress);
		receivedBytes =
		    recvfrom(process->combinedDescriptor, receiveBuffer, sizeof(receiveBuffer), 0,
			     (struct sockaddr *) &peerAddress, &addressLength);
		if (receivedBytes < 0) {
			if (errno == EINTR || errno == EAGAIN) {
				/* Alarm call, i.e. timeout on read. */
				receivedBytes = 0;
				peerAddress.sun_family = AF_INET;	/* dummy */
			} else {
				/* Some other error - report and break. */
				perror(PACKAGE_NAME);
				break;
			}
		}
		if (AF_UNIX == peerAddress.sun_family) {
			const char *leafName = strrchr(peerAddress.sun_path, '/');
			if (NULL != leafName) {
				leafName++;
				if (leafName[0] >= '1' && leafName[0] <= '3') {
					senderIndex = (size_t) (leafName[0] - '1');
				}
			}
		}

		now = time(NULL);
		if (receivedBytes > 0) {
			/*
			 * Record when we last saw any output, so that we
			 * can still keep watching for output after the
			 * process ends (due to buffering etc), but can
			 * eventually time out.
			 *
			 * Note that when we detect the process has ended,
			 * we also set whenOutputLastSeen, in case the
			 * process output something just before exiting and
			 * we've not received it yet - so we wait around a
			 * short while for it.
			 */
			whenOutputLastSeen = now;
		}

		readOffset = 0;
		bytesRemaining = 0;
		if (receivedBytes > 0)
			bytesRemaining = (size_t) receivedBytes;

		/*
		 * If we received a message, add it to the appropriate receive buffer.
		 */
		/*@-mustfreefresh@ */
		/* splint says "newline" leaks memory here, but it doesn't. */
		while (senderIndex < 3 && bytesRemaining > 0) {
			size_t maxBytesToCopy;
			size_t bytesToCopy;
			char *newline;

			maxBytesToCopy = SCW_MAX_READBUFFER - commandOutput[senderIndex].position;
			if (0 == maxBytesToCopy) {
				/* Full buffer - output it now. */
				recordOutput(state, process->settings, process->pid,
					     commandOutput[senderIndex].whichStream,
					     commandOutput[senderIndex].buffer, commandOutput[senderIndex].position);
				commandOutput[senderIndex].position = 0;
				maxBytesToCopy = SCW_MAX_READBUFFER;
			}

			bytesToCopy = bytesRemaining;
			if (bytesToCopy > maxBytesToCopy)
				bytesToCopy = maxBytesToCopy;

			memcpy(commandOutput[senderIndex].buffer + commandOutput[senderIndex].position,	/* flawfinder: ignore */
			       receiveBuffer + readOffset, bytesToCopy);
			/* flawfinder - bounded by calculations above. */
			commandOutput[senderIndex].position += bytesToCopy;
			readOffset += bytesToCopy;
			bytesRemaining -= bytesToCopy;

			/* Output line by line. */
			newline =
			    memchr(commandOutput[senderIndex].buffer, (int) '\n', commandOutput[senderIndex].position);
			while (NULL != newline) {
				size_t newlineOffset = newline - commandOutput[senderIndex].buffer;

				recordOutput(state, process->settings, process->pid,
					     commandOutput[senderIndex].whichStream,
					     commandOutput[senderIndex].buffer, newlineOffset);

				newlineOffset++;
				if (newlineOffset < commandOutput[senderIndex].position) {
					(void) memmove(commandOutput[senderIndex].buffer,
						       &(commandOutput[senderIndex].buffer[newlineOffset]),
						       commandOutput[senderIndex].position - newlineOffset);
					commandOutput[senderIndex].position -= newlineOffset;
				} else {
					commandOutput[senderIndex].position = 0;
				}

				newline = NULL;
				if (commandOutput[senderIndex].position > 0) {
					newline =
					    memchr(commandOutput[senderIndex].buffer, (int) '\n',
						   commandOutput[senderIndex].position);
				}
			}
		}
		/*@+mustfreefresh@ */

		/*
		 * (D4) Terminate the command if MaxRunTime is exceeded.
		 */
		if (commandRunning && process->settings->numMaxRunTime > 0) {
			elapsed = difftime(time(NULL), process->startTime);
			if (elapsed >= (double) (1 + process->settings->numMaxRunTime)) {
				/* 1 second past MaxRunTime, send SIGKILL. */
				debug("%s", "sending SIGKILL");
				(void) kill(process->pid, SIGKILL);
				commandRunning = false;
				whenOutputLastSeen = now;
				process->timedOut = true;
				retcode = SCW_EXIT_RUN_TIMEOUT;
			} else if ((!sentTerm) && elapsed >= (double) (process->settings->numMaxRunTime)) {
				/* First send one SIGTERM. */
				debug("%s", "sending SIGTERM");
				(void) kill(process->pid, SIGTERM);
				whenOutputLastSeen = now;
				sentTerm = true;
				process->timedOut = true;
				retcode = SCW_EXIT_RUN_TIMEOUT;
			}
		}

		/*
		 * Transmit any delayed spools (output lines to transmit to
		 * a URL after a short delay - HTTPInterval).
		 */
		transmitDelayedUrlSpools(state, process->settings);

		/*
		 * Check whether the command process has exited.
		 */

		if (commandRunning) {
			status = 0;
			/*@-type@ */
			/* splint disagreement about __pid_t vs pid_t. */
			waited = waitpid(process->pid, &status, WNOHANG);
			/*@+type@ */
			if (waited < 0 && errno == EINTR) {
				perror(PACKAGE_NAME);
				continue;
			} else if (waited < 0 && errno != EINTR) {
				perror(PACKAGE_NAME);
				process->exitStatus = 1;
				break;
			}

			/*@-predboolint@ */
			/*@-boolops@ */
			/* splint sees WIF...() as non-boolean. */
			if (waited > 0 && WIFEXITED(status)) {
				process->exitStatus = WEXITSTATUS(status);
				commandRunning = false;
				whenOutputLastSeen = now;
				debug("%s: %d", "command exited", process->exitStatus);
			} else if (waited > 0 && WIFSIGNALED(status)) {
				process->exitStatus = 126;
				commandRunning = false;
				whenOutputLastSeen = now;
				debug("%s", "command terminated by signal");
			}
			/*@+predboolint@ *//*@+boolops@ */
		}
	}

	/* Record any incomplete lines still buffered. */
	for (descriptorIndex = 0; descriptorIndex < 3; descriptorIndex++) {
		if (0 == commandOutput[descriptorIndex].position)
			continue;
		recordOutput(state, process->settings, process->pid, commandOutput[descriptorIndex].whichStream,
			     commandOutput[descriptorIndex].buffer, commandOutput[descriptorIndex].position);
	}

	/* Close the command's receive socket. */
	if (process->combinedDescriptor >= 0) {
		(void) close(process->combinedDescriptor);
		process->combinedDescriptor = -1;
	}

	/* Remove the command's Unix sockets and working directory. */
	if (NULL != process->workDir) {
		char socketPath[1024];	 /* flawfinder: ignore */
		/* flawfinder - we bound calls to the buffer and zero it. */

		memset(socketPath, 0, sizeof(socketPath));
		(void) snprintf(socketPath, sizeof(socketPath), "%s/receiver", process->workDir);
		(void) unlink(socketPath);
		(void) snprintf(socketPath, sizeof(socketPath), "%s/1", process->workDir);
		(void) unlink(socketPath);
		(void) snprintf(socketPath, sizeof(socketPath), "%s/2", process->workDir);
		(void) unlink(socketPath);
		(void) snprintf(socketPath, sizeof(socketPath), "%s/3", process->workDir);
		(void) unlink(socketPath);
		(void) rmdir(process->workDir);
		process->workDir = NULL;
	}

	return retcode;
}
