/*
 * Functions for enumerating users and items.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <glob.h>
#include <errno.h>

struct arrayInfo {
	/*@null@ */
	/*@owned@ */
	char **array;			 /* array of string pointers */
	size_t memberCount;		 /* number of items stored in array */
	size_t allocated;		 /* maximum number of items that can be stored */
};


/*
 * Given a string, allocate a copy of it in which the "{USER}" placeholder
 * is replaced with "*", and populate *prefixLength and *suffixLength with
 * the length of the strings preceding and succeeding the placeholder.
 *
 * The returned pointer is part of state->strings and so should not be
 * passed to free(), as clearState() will free it later.
 *
 * Returns NULL on error.
 */
/*@null@ */
/*@dependent @*/
static char *convertToGlob(struct scwState *state, const char *string, size_t length, size_t *prefixLength,
			   size_t *suffixLength)
{
	char *globVersion = NULL;
	size_t readPosition = 0, writePosition = 0;
	bool userPlaceholderFound = false;

	/*
	 * "*" is 5 bytes shorter than "{USER}", but we might not encounter
	 * {USER} at all, so allocate the same amount of space as the
	 * original string.
	 */
	globVersion = stringCopy(state, NULL, length);
	if (NULL == globVersion)
		return NULL;

	while (readPosition < length && writePosition < length) {
		size_t remainingLength;

		if ('{' != string[readPosition]) {
			globVersion[writePosition] = string[readPosition];
			readPosition++;
			writePosition++;
			continue;
		}
		remainingLength = length - readPosition;
		/*@-unrecog@ *//* splint doesn't know about strncasecmp(). */
		if (userPlaceholderFound || remainingLength < 6
		    || 0 != strncasecmp(string + readPosition, STATIC_STRING("{USER}"))) {
			globVersion[writePosition] = string[readPosition];
			readPosition++;
			writePosition++;
			continue;
		}
		/*@+unrecog@ */

		userPlaceholderFound = true;
		*prefixLength = writePosition;
		globVersion[writePosition] = '*';
		readPosition += 6;	    /* strlen("{USER}") */
		writePosition++;
		*suffixLength = length - readPosition;	/* not writePosition */
		/* not writePosition because we've skipped over {USER}. */
	}

	debug("[%.*s] -> [%s], %s=%d, %s=%d", length, string, globVersion, "prefixLength", *prefixLength,
	      "suffixLength", *suffixLength);

	return globVersion;
}


/*
 * Add a copy of the string to the array, resizing the array as necessary,
 * if the string isn't already listed in the array.
 */
static void addStringToArray(struct scwState *state, const char *string, size_t length, struct arrayInfo *arrayInfo)
{
	size_t arrayIndex;
	char *copiedString;

	/*
	 * If the array isn't empty, check the string isn't already in it.
	 *
	 * Note we could use a binary tree or something to make this more
	 * efficient, but it's not worth the complexity for something that's
	 * going to be called a few dozen times at most.  It won't scale
	 * well if there are thousands of usernames.
	 */
	for (arrayIndex = 0; NULL != arrayInfo->array && arrayIndex < arrayInfo->memberCount; arrayIndex++) {
		const char *member = arrayInfo->array[arrayIndex];
		if (NULL == member)
			continue;
		if (length != strlen(member))	/* flawfinder: ignore */
			continue;
		/*
		 * flawfinder warns about strlen() and non null-terminated
		 * strings, but the array contains only null-terminated
		 * strings that we placed there, so it's OK.
		 */
		if (0 == strncmp(string, member, length)) {
			/* Already in the array - early return. */
			return;
		}
	}

	/* Extend the array if it's not big enough. */
	/*@-usereleased@ *//* splint doesn't like realloc() here. */
	if (arrayInfo->memberCount >= arrayInfo->allocated) {
		size_t newAllocationCount = 100 + arrayInfo->allocated;
		char **newArray;
		newArray = realloc(arrayInfo->array, newAllocationCount * sizeof(char *));
		if (NULL == newArray) {
			perror(PACKAGE_NAME);
#if SPLINT
			arrayInfo->array = arrayInfo->array;	/* for splint */
#endif
			return;
		}
		arrayInfo->array = newArray;
		arrayInfo->allocated = newAllocationCount;
	}
	/*@+usereleased@ */

	if (NULL == arrayInfo->array)
		return;

	/* Add a copy of the string to the array. */
	copiedString = stringCopy(state, string, length);
	if (NULL == copiedString)
		return;
	/*@-dependenttrans@ */
	arrayInfo->array[arrayInfo->memberCount] = copiedString;
	/* splint doesn't know that array members are treated as dependent. */
	/*@+dependenttrans@ */
	debug("%s[%d]=[%s]", "array", arrayInfo->memberCount, copiedString);

	arrayInfo->memberCount++;
}


/*
 * Expand a glob pattern, and from the results, strip the prefix and the
 * suffix to get just the username portion of the path, and add each
 * username to the array.
 */
/*@-compdestroy@ */
/*
 * splint doesn't see globfree() as deallocating all of globResult, so
 * reports memory leaks here unless we turn off the compdestroy check.
 */
static void findUsersByGlob(struct scwState *state, const char *globPattern, size_t prefixLength, size_t suffixLength,
			    struct arrayInfo *arrayInfo)
{
	glob_t globResult;
	size_t pathIndex;

	memset(&globResult, GLOB_NOSORT | GLOB_NOESCAPE, sizeof(globResult));
	/*@-nullpass@ *//* we may pass NULL to glob(). */
	if (0 != glob(globPattern, 0, NULL, &globResult)) {
		globfree(&globResult);
		return;
	}
	/*@+nullpass@ */

	for (pathIndex = 0; pathIndex < (size_t) (globResult.gl_pathc); pathIndex++) {
		const char *path = globResult.gl_pathv[pathIndex];
		size_t pathLength;
		const char *usernameMatch;
		size_t matchLength;

		if (NULL == path)
			continue;
		pathLength = strlen(path);  /* flawfinder: ignore */
		/* flawfinder - glob() null-terminates the strings it returns. */

		if (pathLength < prefixLength)
			continue;

		/* Skip the prefix. */
		usernameMatch = path + prefixLength;
		matchLength = pathLength - prefixLength;

		/* Check anything would be left after suffix removal. */
		if (matchLength <= suffixLength)
			continue;

		matchLength -= suffixLength;

		debug("%s: [%.*s]", "username found", matchLength, usernameMatch);
		addStringToArray(state, usernameMatch, matchLength, arrayInfo);
	}

	globfree(&globResult);
}

/*@+compdestroy@ */


/*
 * Replace {USER} in the global UserConfigFile value with * and use it as a
 * glob pattern to find user config files.  Add the usernames found this way
 * to the array.
 */
static void enumerateUsersByUserConfigFile(struct scwState *state, struct arrayInfo *arrayInfo)
{
	char *userConfigFile = NULL;
	size_t prefixLength = 0, suffixLength = 0;

	if (NULL == state->globalSettings.userConfigFile.rawValue) {
		/*@-mustfreefresh@ *//* gettext causes a splint warning. */
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, "UserConfigFile", _("no value set"));
		return;
		/*@+mustfreefresh@ */
	}

	userConfigFile =
	    convertToGlob(state, state->globalSettings.userConfigFile.rawValue,
			  state->globalSettings.userConfigFile.rawLength, &prefixLength, &suffixLength);
	if (NULL == userConfigFile)
		return;

	findUsersByGlob(state, userConfigFile, prefixLength, suffixLength, arrayInfo);
}


/*
 * Replace {USER} in the global ItemsDir value with * and use it as a glob
 * pattern to find user item directories.  Add the usernames found this way
 * to the array.
 */
static void enumerateUsersByItemsDir(struct scwState *state, struct arrayInfo *arrayInfo)
{
	char *itemsDir = NULL;
	size_t prefixLength = 0, suffixLength = 0;

	if (NULL == state->globalSettings.itemsDir.rawValue) {
		/*@-mustfreefresh@ *//* gettext causes a splint warning. */
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, "ItemsDir", _("no value set"));
		return;
		/*@+mustfreefresh@ */
	}

	itemsDir =
	    convertToGlob(state, state->globalSettings.itemsDir.rawValue,
			  state->globalSettings.itemsDir.rawLength, &prefixLength, &suffixLength);
	if (NULL == itemsDir)
		return;

	findUsersByGlob(state, itemsDir, prefixLength, suffixLength, arrayInfo);
}


/*
 * Enumerate all users known to the system, by first looking for files
 * matching the global UserConfigFile setting when {USER} is replaced with
 * "*" and expanded as a glob pattern, and then also by looking for
 * directories matching the global ItemsDir where the same glob strategy is
 * employed.
 *
 * Populates *usernameArrayPointer with an array of pointers to usernames,
 * and *usernameCountPointer with the number of usernames.  The array, and
 * the usernames within it, are allocated in the main state->strings array
 * so shouldn't be passed to free() - they will be freed by clearState().
 */
void enumerateAllUsers(struct scwState *state, char ***usernameArrayPointer, size_t *usernameCountPointer)
{
	struct arrayInfo arrayInfo;

	arrayInfo.array = NULL;
	arrayInfo.memberCount = 0;
	arrayInfo.allocated = 0;

	*usernameCountPointer = 0;

	enumerateUsersByUserConfigFile(state, &arrayInfo);
	enumerateUsersByItemsDir(state, &arrayInfo);

	/* Allocate a new array from state->strings. */
	if (arrayInfo.memberCount > 0 && NULL != arrayInfo.array) {
		size_t arrayBytes = arrayInfo.memberCount * sizeof(char *);
		char **newUsernameArray;

		/* NB the "-1" is because stringCopy() adds 1 to the length for a null terminator. */
		newUsernameArray = (char **) stringCopy(state, NULL, arrayBytes - 1);
		if (NULL == newUsernameArray)
			return;
		memcpy(newUsernameArray, arrayInfo.array, arrayBytes);	/* flawfinder: ignore */
		/* flawfinder - the buffer is allocated to be this exact size. */
		/*@-dependenttrans@ */
		/* we have no way to tell splint that the username array is dependent. */
		*usernameArrayPointer = newUsernameArray;
		/*@+dependenttrans@ */
		*usernameCountPointer = arrayInfo.memberCount;
	}

	if (NULL != arrayInfo.array)
		free(arrayInfo.array);
}


/*
 * Filter for scandir(), returning nonzero if the directory entry refers to
 * an item-related file (suffix ".cf", ".sh", or ".pl").
 */
static int itemFilter(const struct dirent *dirEntry)
{
	const char *name;
	size_t nameLength;

	if (NULL == dirEntry)
		return 0;
	name = dirEntry->d_name;
	if (NULL == name)
		return 0;

	nameLength = strlen(name);	    /* flawfinder: ignore */
	/* flawfinder - scandir() null-terminates directory entry names. */

	/* Check for the right filename suffix. */
	if (nameLength < 4)
		return 0;
	if ('.' != name[nameLength - 3])
		return 0;
	if ((0 != strncmp(name + nameLength - 2, "cf", 2)) && (0 != strncmp(name + nameLength - 2, "sh", 2))
	    && (0 != strncmp(name + nameLength - 2, "pl", 2)))
		return 0;

	/* Check it's a valid item name. */
	if (!validateItemName(name, nameLength - 3))
		return 0;

	return 1;
}


/*
 * Enumerate items for the current user, by looking in ItemsDir.
 *
 * Populates *itemArrayPointer with an array of pointers to item names, and
 * *itemCountPointer with the number of items.  The array, and the item
 * names within it, are allocated in the main state->strings array so
 * shouldn't be passed to free() - they will be freed by clearState().
 *
 * Note that as a side effect, this clears state->itemSettings.
 */
void enumerateUserItems(struct scwState *state, char ***itemArrayPointer, size_t *itemCountPointer)
{
	struct arrayInfo arrayInfo;
	struct scwSettings combinedSettings;
	struct dirent **directoryEntries = NULL;
	int fileCount;
	size_t fileIndex;

	arrayInfo.array = NULL;
	arrayInfo.memberCount = 0;
	arrayInfo.allocated = 0;

	*itemCountPointer = 0;

	/* Clear itemSettings so it doesn't interfere with combineSettings(). */
	memset(&(state->itemSettings), 0, sizeof(state->itemSettings));

	/* Combine the global and user settings, and expand ItemsDir. */
	memset(&combinedSettings, 0, sizeof(combinedSettings));
	if (0 != combineSettings(state, &combinedSettings))
		return;
	expandRawValue(state, &(combinedSettings.itemsDir));
	if (NULL == combinedSettings.itemsDir.expandedValue)
		return;

	/* List the contents of ItemsDir. */
	fileCount = scandir(combinedSettings.itemsDir.expandedValue, &directoryEntries, itemFilter, alphasort);
	if (fileCount < 0) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, combinedSettings.itemsDir.expandedValue, strerror(errno));
		return;
	}

	/* Add each item file, without its suffix, to the array. */
	for (fileIndex = 0; NULL != directoryEntries && fileIndex < (size_t) fileCount; fileIndex++) {
		/*@dependent@ */ const char *name;
		size_t nameLength;

		name = directoryEntries[fileIndex]->d_name;
		if (NULL == name)
			continue;
		nameLength = strlen(name);  /* flawfinder: ignore */
		/* flawfinder - scandir() null-terminates directory entry names. */
		if (nameLength < 4)
			continue;
		nameLength -= 3;	    /* remove suffix */

		debug("%s: [%.*s]", "item found", nameLength, name);
		addStringToArray(state, name, nameLength, &arrayInfo);

		free(directoryEntries[fileIndex]);
	}
	free(directoryEntries);

	/* Allocate a new array from state->strings. */
	if (arrayInfo.memberCount > 0 && NULL != arrayInfo.array) {
		size_t arrayBytes = arrayInfo.memberCount * sizeof(char *);
		char **newItemArray;

		/* NB the "-1" is because stringCopy() adds 1 to the length for a null terminator. */
		newItemArray = (char **) stringCopy(state, NULL, arrayBytes - 1);
		if (NULL == newItemArray)
			return;
		memcpy(newItemArray, arrayInfo.array, arrayBytes);	/* flawfinder: ignore */
		/* flawfinder - the buffer is allocated to be this exact size. */
		/*@-dependenttrans@ */
		/* we have no way to tell splint that the item array is dependent. */
		*itemArrayPointer = newItemArray;
		/*@+dependenttrans@ */
		*itemCountPointer = arrayInfo.memberCount;
	}

	if (NULL != arrayInfo.array)
		free(arrayInfo.array);
}
