/*
 * Functions for file handling.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <utime.h>
#include <errno.h>


/*
 * Open the file "filename" for reading, checking that it is a regular file
 * rather than a device, and if possible, that the last component of its
 * path is not a symbolic link.  Returns -1 on error.
 */
int fileOpenForRead(const char *filename)
{
	int fd;
	struct stat statBuf;

	fd = open(filename, O_RDONLY	    /* flawfinder: ignore */
#ifdef O_NOFOLLOW
		  | O_NOFOLLOW
#endif
	    );
	if (fd < 0)
		return -1;

	/*
	 * flawfinder highlights that open() needs to check that an attacker
	 * can't misuse symlinks, point to a device file, cause a race
	 * condition, control its ancestors, or change its contents.  This
	 * function's purpose is to check the first two.  The caller is
	 * responsible for the rest.  So we silence this warning.
	 */

	if ((0 != fstat(fd, &statBuf)) || (!S_ISREG((mode_t) (statBuf.st_mode)))) {
		int oldError;
		oldError = errno;
		(void) close(fd);
		errno = oldError;
		return -1;
	}

	return fd;
}


/*
 * Open the file "filename" in the same way as fileOpenForRead() but return
 * a file stream rather than a file descriptor, or NULL on error.
 */
/*@-dependenttrans@*/
/*@null@*/
/*@dependent@*/
FILE *fileOpenStreamForRead(const char *filename)
{
	int fd;
	FILE *stream;

	fd = fileOpenForRead(filename);
	if (fd < 0)
		return NULL;

	stream = fdopen(fd, "r");
	if (NULL == stream) {
		int oldError;
		oldError = errno;
		(void) close(fd);
		errno = oldError;
		return NULL;
	}

	return stream;
}

/*@+dependenttrans@*/


/*
 * If the given directory does not exist, create it.  Returns a nonzero
 * SCW_EXIT_* status if the directory doesn't exist and creation failed,
 * after reporting the error on stderr.
 */
int directoryCreate(const char *path, size_t pathLength)
{
	char *pathCopy;
	struct stat statBuf;

	/* Strip trailing / characters. */
	while (pathLength > 1 && '/' == path[pathLength - 1])
		pathLength--;

	if (pathLength < 1)
		return 0;

	/* Make a temporary null-terminated copy of the path. */
	pathCopy = malloc(pathLength + 1);
	if (NULL == pathCopy) {
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, strerror(errno));
		return SCW_EXIT_ERROR;
	}
	memset(pathCopy, 0, pathLength + 1);
	memcpy(pathCopy, path, pathLength); /* flawfinder: ignore */
	/*
	 * flawfinder warns that memcpy() doesn't know the destination
	 * buffer size.  We have explicitly allocated the buffer to be large
	 * enough so there is no risk here.
	 */

	if (0 == stat(pathCopy, &statBuf) && S_ISDIR((mode_t) (statBuf.st_mode))) {
		debug("%s: %s", pathCopy, "OK - already exists");
		free(pathCopy);
		return 0;
	}

	if (0 != mkdir(pathCopy, 0755)) {
		int oldError;
		oldError = errno;
		debug("%s: %s: %s", pathCopy, "mkdir failed", strerror(errno));
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, pathCopy, strerror(errno));
		free(pathCopy);
		errno = oldError;
		return SCW_EXIT_ERROR;
	}

	debug("%s: %s", pathCopy, "mkdir succeeded");

	free(pathCopy);
	return 0;
}


/*
 * Create the parent directory of a path, if it does not already exist.
 *
 * Returns a nonzero SCW_EXIT_* status if there was an error, after
 * reporting it on stderr.
 */
int directoryParentCreate(const char *path, size_t pathLength)
{
	/* Strip trailing / characters. */
	while (pathLength > 1 && '/' == path[pathLength - 1])
		pathLength--;

	/* Move back to the preceding /. */
	while (pathLength > 1 && path[pathLength - 1] != '/')
		pathLength--;

	if (pathLength < 1)
		return 0;

	return directoryCreate(path, pathLength);
}


/*
 * Allocate a buffer and populate it with directory + "/" + filename and a
 * terminating null byte.  The buffer should be passed to free() as soon as
 * it is no longer needed.
 *
 * Returns NULL on error, and reports it on stderr.
 */
/*@null@*/
static char *allocateFullPath(const char *directory, size_t directoryLength, const char *filename,
			      size_t filenameLength)
{
	char *fullPath;
	size_t fullPathLength;

	/* Full path is directory + "/" + filename. */
	fullPathLength = directoryLength + 1 + filenameLength;

	/* Allocated space must include a terminating null byte. */
	fullPath = malloc(fullPathLength + 1);
	if (NULL == fullPath) {
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, strerror(errno));
		return NULL;
	}
	/* Clear the full area, including the terminator. */
	memset(fullPath, 0, fullPathLength + 1);

	/* Populate the buffer with directory + "/" + filename. */
	memcpy(fullPath, directory, directoryLength);	/* flawfinder: ignore */
	fullPath[directoryLength] = '/';
	memmove(fullPath + directoryLength + 1, filename, filenameLength);
	/*
	 * flawfinder warns that memcpy() doesn't know the destination
	 * buffer size.  We have explicitly allocated the buffer to be large
	 * enough so there is no risk here.  The target pointer here is the
	 * allocated space plus the length of directory and "/", we're
	 * copying the filename there, and we know we allocated space for
	 * exactly that amount plus a terminating null byte.
	 */

	return fullPath;
}


/*
 * Create a file under a directory if the file does not already exist.  If
 * it does exist, update its last-modification time only if
 * "alwaysSetModified" is true.
 *
 * Returns a nonzero SCW_EXIT_* status if there was an error, after
 * reporting it on stderr.
 */
int fileCreateAt(const char *directory, size_t directoryLength, const char *filename, size_t filenameLength,
		 bool alwaysSetModified)
{
	char *fullPath;
	int fd;

	/* Make a temporary null-terminated copy of the full path. */

	fullPath = allocateFullPath(directory, directoryLength, filename, filenameLength);
	if (NULL == fullPath)
		return SCW_EXIT_ERROR;

	if (0 == access(fullPath, F_OK)) {  /* flawfinder: ignore */
		/*
		 * flawfinder suggests that access() usually indicates a
		 * race condition.  Here we're only using it to determine
		 * whether we need to create the file, so all an adversary
		 * could achieve here by moving or redirecting the target
		 * file would be to cause us to create a new file (with a
		 * new timestamp) incorrectly, which they could already do
		 * themselves if they had the ability to move the target
		 * file.
		 */
		/* If the file already exists, it doesn't need to be created. */
		if (!alwaysSetModified) {
			/* Return if we don't need to set the last-modified time. */
			debug("%s: %s", fullPath, "already exists and no utime needed");
			free(fullPath);
			return 0;
		}
		/* Set the last-modified time. */
		if (0 != utime(fullPath, NULL)) {
			int oldError;
			oldError = errno;
			debug("%s: %s: %s", fullPath, "utime failed", strerror(errno));
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, fullPath, strerror(errno));
			free(fullPath);
			errno = oldError;
			return SCW_EXIT_ERROR;
		}
		debug("%s: %s", fullPath, "already exists and utime successful");
		free(fullPath);
		return 0;
	}

	/* The file needs to be created. */

	fd = open(fullPath, O_WRONLY | O_CREAT | O_EXCL, 0644);	/* flawfinder: ignore */
	/*
	 * flawfinder warns of CWE-362 with open().  This is mitigated by
	 * O_EXCL, which does not follow symlinks, and by the fact that we
	 * are not using O_TRUNC or writing to the file at all, so the only
	 * effect is to create a new file with a new timestamp.
	 */
	if (fd < 0) {
		int oldError;
		oldError = errno;
		debug("%s: %s: %s", fullPath, "open failed", strerror(errno));
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, fullPath, strerror(errno));
		free(fullPath);
		errno = oldError;
		return SCW_EXIT_ERROR;
	}
	(void) close(fd);

	debug("%s: %s", fullPath, "created successfully");

	free(fullPath);
	return 0;
}


/*
 * Delete a file under a directory.
 *
 * Returns a nonzero SCW_EXIT_* status if there was an error, after
 * reporting it on stderr.
 */
int fileDeleteAt(const char *directory, size_t directoryLength, const char *filename, size_t filenameLength)
{
	char *fullPath;

	/* Make a temporary null-terminated copy of the full path. */

	fullPath = allocateFullPath(directory, directoryLength, filename, filenameLength);
	if (NULL == fullPath)
		return SCW_EXIT_ERROR;

	if (0 != access(fullPath, F_OK)) {  /* flawfinder: ignore */
		/*
		 * flawfinder's warning about access() relates to race
		 * conditions.  Here we're only using it so we can return
		 * without error if the file is already gone, rather than
		 * potentially returning an error from unlink() being called
		 * below on a nonexistent file, so there is no risk.
		 */
		debug("%s: %s", fullPath, "does not exist, no removal needed");
		free(fullPath);
		return 0;
	}

	if (0 != unlink(fullPath)) {
		int oldError;
		oldError = errno;
		debug("%s: %s: %s", fullPath, "unlink failed", strerror(errno));
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, fullPath, strerror(errno));
		free(fullPath);
		errno = oldError;
		return SCW_EXIT_ERROR;
	}

	debug("%s: %s", fullPath, "removed successfully");

	free(fullPath);
	return 0;
}


/*
 * Open a file in append mode, returning a file descriptor on success.  On
 * error, -1 is returned, and the error is reported on stderr.
 *
 * NB debugWriteOutput() calls this, so this function must not call debug().
 */
int fileOpenForAppend(const char *filename)
{
	int fd;

	fd = open(filename, O_WRONLY	    /* flawfinder: ignore */
#ifdef O_NOFOLLOW
		  | O_NOFOLLOW
#endif
		  | O_APPEND | O_CREAT, 0644);

	/*
	 * flawfinder highlights that a user of open() needs to check that
	 * an attacker can't misuse symlinks, point to a device file, cause
	 * a race condition, control its ancestors, or change its contents. 
	 * We are opening in append-only mode so the risk is minimal and is
	 * further reduced by the use of O_NOFOLLOW.
	 */

	if (fd < 0) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, filename, strerror(errno));
		return -1;
	}

	return fd;
}


/*
 * Open a file under a directory, in append mode, returning a file
 * descriptor on success.  On error, -1 is returned, and the error is
 * reported on stderr.
 */
int fileOpenForAppendAt(const char *directory, size_t directoryLength, const char *filename, size_t filenameLength)
{
	char *fullPath;
	int fd, oldError;

	/* Make a temporary null-terminated copy of the full path. */

	fullPath = allocateFullPath(directory, directoryLength, filename, filenameLength);
	if (NULL == fullPath)
		return -1;

	fd = fileOpenForAppend(fullPath);

	oldError = errno;
	free(fullPath);
	errno = oldError;

	return fd;
}


/*
 * Replace a file's contents atomically, under a directory, by creating and
 * populating a new temporary file, and renaming it over the original.
 *
 * Returns a nonzero SCW_EXIT_* status if there was an error, after
 * reporting the error on stderr.
 */
int fileReplaceContentsAt(const char *directory, size_t directoryLength, const char *filename, size_t filenameLength,
			  const char *format, ...)
{
	char *targetPath;
	size_t targetPathLength;
	char *tempPath;
	size_t tempPathLength;
	int fd;
	FILE *stream;
	va_list varArgs;
	bool failed;

	/* Make a temporary null-terminated copy of the full path. */

	targetPath = allocateFullPath(directory, directoryLength, filename, filenameLength);
	if (NULL == targetPath)
		return SCW_EXIT_ERROR;

	/* Buffer for the temporary filename. */

	/* Target path is directory + "/" + filename. */
	targetPathLength = directoryLength + 1 + filenameLength;

	/* Temporary filename is target path plus ".XXXXXX". */
	tempPathLength = targetPathLength + 7;

	/* Allocated space must include a terminating null byte. */
	tempPath = malloc(tempPathLength + 1);
	if (NULL == tempPath) {
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, strerror(errno));
		free(targetPath);
		return SCW_EXIT_ERROR;
	}
	/* Clear the full area, including the terminator. */
	memset(tempPath, 0, tempPathLength + 1);

	/* Populate the buffer with the target path plus the temp file template. */
	memcpy(tempPath, targetPath, targetPathLength);	/* flawfinder: ignore */
	memcpy(tempPath + targetPathLength, ".XXXXXX", 7);	/* flawfinder: ignore */
	/* flawfinder - see above. */

	/* Create the temporary file and set its permissions. */

	/*@-unrecog@ */
	/* splint doesn't know about mkstemp(). */
	fd = mkstemp(tempPath);		    /* flawfinder: ignore */
	/*@+unrecog@ */
	/*
	 * flawfinder warns about CWE-377, relating to mkstemp() not always
	 * using the right file mode and open() flags.  This is mitigated
	 * here by the fact that we're not operating in the world-writable
	 * system temporary directory, but in a more controlled area.
	 */
	if (fd < 0) {
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, strerror(errno));
		free(tempPath);
		free(targetPath);
		return SCW_EXIT_ERROR;
	}

	if (fchmod(fd, 0644) < 0) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, tempPath, strerror(errno));
		(void) close(fd);
		(void) unlink(tempPath);
		free(tempPath);
		free(targetPath);
		return SCW_EXIT_ERROR;
	}

	/* Open a file stream and write the contents to it. */

	stream = fdopen(fd, "w");
	if (NULL == stream) {
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, strerror(errno));
		(void) close(fd);
		(void) unlink(tempPath);
		free(tempPath);
		free(targetPath);
		return SCW_EXIT_ERROR;
	}

	failed = false;
	va_start(varArgs, format);
	if (vfprintf(stream, format, varArgs) < 0) {	/* flawfinder: ignore */
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, tempPath, strerror(errno));
		failed = true;
	}
	/*
	 * flawfinder - the format string is variable by design, it is up to
	 * the caller to ensure it makes sense with the other arguments
	 * provided.
	 */
	va_end(varArgs);

	/* Close the file stream, and with it the temp file descriptor. */
	(void) fclose(stream);
	/* not needed because fclose() does it: (void) close(fd); */

	/* Replace the target file with the temporary file. */

	if (!failed) {
		if (rename(tempPath, targetPath) < 0) {
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, targetPath, strerror(errno));
			failed = true;
		}
	}

	/* Remove the temporary file in case it was left over. */
	(void) unlink(tempPath);

	/* Free the filename buffers. */
	free(tempPath);
	free(targetPath);

	if (failed) {
		return SCW_EXIT_ERROR;
	}

	return 0;
}


/*
 * Return true if the file exists under the directory.
 */
bool fileExistsAt(const char *directory, size_t directoryLength, const char *filename, size_t filenameLength)
{
	char *fullPath;
	bool fileExists;

	/* Make a temporary null-terminated copy of the full path. */

	fullPath = allocateFullPath(directory, directoryLength, filename, filenameLength);
	if (NULL == fullPath)
		return false;

	/* Check whether the file exists. */
	fileExists = false;
	if (0 == access(fullPath, F_OK)) {  /* flawfinder: ignore */
		fileExists = true;
	}
	/*
	 * flawfinder warns of CWE-362/CWE-367 when using access() before
	 * open(), but in general we're not opening the file we're checking
	 * with this function.
	 */

	free(fullPath);
	return fileExists;
}


/*
 * Return the result of stat() on the file under the directory, populating
 * statBuf in the process.
 */
int fileStatAt(const char *directory, size_t directoryLength, const char *filename, size_t filenameLength,
	       struct stat *statBuf)
{
	char *fullPath;
	int retcode, oldError;

	/* Make a temporary null-terminated copy of the full path. */

	fullPath = allocateFullPath(directory, directoryLength, filename, filenameLength);
	if (NULL == fullPath)
		return -1;

	retcode = stat(fullPath, statBuf);

	oldError = errno;
	debug("%s(%s) = %d", "stat", fullPath, retcode);
	free(fullPath);
	errno = oldError;

	return retcode;
}


/*
 * Safely open a file under a directory for reading, returning a file stream
 * pointer, or NULL on error (errors are not reported on stderr).
 */
/*@null@*/
/*@dependent@*/
FILE *fileOpenStreamForReadAt(const char *directory, size_t directoryLength, const char *filename,
			      size_t filenameLength)
{
	FILE *stream;
	char *fullPath;
	int oldError;

	/* Make a temporary null-terminated copy of the full path. */

	fullPath = allocateFullPath(directory, directoryLength, filename, filenameLength);
	if (NULL == fullPath)
		return NULL;

	/* Open the file and free the path string, preserving errno. */

	stream = fileOpenStreamForRead(fullPath);

	oldError = errno;
	free(fullPath);
	errno = oldError;

	return stream;
}


/*
 * Open a file under a directory, read an integer from it, and return the
 * integer, or -1 if nothing could be read.
 */
int fileReadIntegerFromFileAt(const char *directory, size_t directoryLength, const char *filename,
			      size_t filenameLength)
{
	FILE *stream;
	int valueFromFile, oldError;

	stream = fileOpenStreamForReadAt(directory, directoryLength, filename, filenameLength);
	if (NULL == stream)
		return -1;

	if (fscanf(stream, "%d", &valueFromFile) != 1) {
		oldError = errno;
		(void) fclose(stream);
		errno = oldError;
		return -1;
	}

	(void) fclose(stream);
	return valueFromFile;
}


/*
 * Write a complete buffer to a file descriptor, returning false on failure.
 */
bool fileWriteBuffer(int fd, const char *buffer, size_t length)
{
	size_t offset = 0;
	size_t remaining = length;

	while (remaining > 0) {
		ssize_t bytesWritten;

		bytesWritten = write(fd, buffer + offset, remaining);

		if (bytesWritten < 0) {
			if ((EINTR == errno) || (EAGAIN == errno)) {
				continue;
			}
			return false;
		}
		if (bytesWritten < 1)
			return false;

		remaining -= bytesWritten;
		offset += bytesWritten;
	}

	return true;
}


/*
 * Create a temporary file in the system temporary directory, returning
 * false on failure.  Populates *tempFilenamePointer with an allocated
 * string containing its full path, and tempDescriptorPointer with a
 * read-write file descriptor.  The caller must not free the string.
 */
bool tempFileCreate(struct scwState *state, nullable_string_t * tempFilenamePointer, int *tempDescriptorPointer)
{
	char *tempDirectory;
	size_t tempDirectoryLength;
	const char *templateSuffix = "scw.XXXXXX";
	size_t templateSuffixLength = strlen(templateSuffix);	/* flawfinder: ignore */
	char *tempPath = NULL;
	size_t tempPathLength;
	int tempDescriptor;
	mode_t oldMode;

	/* flawfinder - strlen() on static string is OK. */

	/* Construct a temporary filename template. */

	tempDirectory = (char *) getenv("TMPDIR");	/* flawfinder: ignore */
	if ((NULL == tempDirectory) || ('\0' == tempDirectory[0]))
		tempDirectory = (char *) getenv("TMP");	/* flawfinder: ignore */
	if ((NULL == tempDirectory) || ('\0' == tempDirectory[0]))
		tempDirectory = "/tmp";
	tempDirectoryLength = strlen(tempDirectory);	/* flawfinder: ignore */

	/*
	 * flawfinder rationale: null and zero-size values of $TMPDIR and
	 * $TMP are rejected, and the destination buffer is sized
	 * appropriately.
	 */

	tempPathLength = tempDirectoryLength + 1 + templateSuffixLength;
	tempPath = stringCopy(state, NULL, tempPathLength);
	if (NULL == tempPath) {
		/* Memory allocation failure - fatal error, immediate return. */
		perror(PACKAGE_NAME);
		return false;
	}
	memcpy(tempPath, tempDirectory, tempDirectoryLength);	/* flawfinder: ignore */
	tempPath[tempDirectoryLength] = '/';
	memcpy(tempPath + tempDirectoryLength + 1, templateSuffix, templateSuffixLength);	/* flawfinder: ignore */
	/* flawfinder - these buffers are sized appropriately. */

	/* Convert the template into a temporary file. */

	/*@-type@ *//* splint disagrees that mode_t == __mode_t */
	oldMode = umask(0077);		    /* flawfinder: ignore */
	/*@-unrecog@ *//* splint doesn't know mkstemp() */
	tempDescriptor = mkstemp(tempPath); /* flawfinder: ignore */
	/*@+unrecog@ */
	(void) umask(oldMode);		    /* flawfinder: ignore */
	/*@+type@ */

	/*
	 * flawfinder - the umask is being temporarily set here to mitigate
	 * the risk of too-permissive mkstemp() defaults.
	 */

	if (tempDescriptor < 0) {
		perror(PACKAGE_NAME);
		return false;
	}

	if (NULL != tempDescriptorPointer)
		*tempDescriptorPointer = tempDescriptor;

	/*@-dependenttrans@ */
	if (NULL != tempFilenamePointer)
		*tempFilenamePointer = tempPath;
	/*@+dependenttrans@ */

	return true;
}
