/*
 * Output command-line help to stdout.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"


/*
 * Structure holding the displayed descriptions of each option and action -
 * the short option such as "-s" (or action name like "run"), the option's
 * long counterpart such as "--size", the name of its argument such as
 * "SIZE", and the description of what the option or action does.  Any of
 * them may be NULL.
 *
 * In the initialiser for this structure, translatable strings are wrapped
 * with N_() to indicate that they are to be translated into the operator's
 * language by this function at run-time.
 *
 * Note that "shortOption" and "longOption" should never be marked as
 * translatable, as they must remain consistent across all locales.
 *
 * The list is terminated with a NULL shortOption value - to leave a gap, set
 * shortOption to an empty string instead.
 */
struct scwParameterDefinition {
	/*@null@ */ const char *shortOption;
	/*@null@ */ const char *longOption;
	/*@null@ *//*@observer@ */ const char *parameterArgument;
	/*@null@ *//*@observer@ */ const char *description;
	struct {
		/*
		 * Structure holding the width, in display character
		 * positions, of the individual parts of the description of
		 * each parameter - calculated after initialisation, so that
		 * translation can be performed.
		 */
		size_t shortOption;
		size_t longOption;
		size_t parameterArgument;
		size_t description;
	} width;
};


/*
 * Display command-line help.
 */
void displayHelp(void)
{
	struct scwParameterDefinition parameterDefinitions[] = {
		{ "-c", "--config", N_("FILE"),
		 N_("read global configuration from FILE"),
		 { 0, 0, 0, 0} },
		{ "-s", "--set", N_("SETTING=VALUE"),
		 N_("override an item or configuration setting"),
		 { 0, 0, 0, 0} },
		{ "-f", "--force", NULL,
		 N_("run the item even if it is disabled"),
		 { 0, 0, 0, 0} },
		{ "-S", "--strict", NULL,
		 N_("stop if locks, logs, or metrics are not writable"),
		 { 0, 0, 0, 0} },
		{ "-e", "--enabled", NULL,
		 N_("only list enabled items"),
		 { 0, 0, 0, 0} },
		{ "-d", "--disabled", NULL,
		 N_("only list disabled items"),
		 { 0, 0, 0, 0} },
		{ "-i", "--info", NULL,
		 N_("show extra information about each item"),
		 { 0, 0, 0, 0} },
		{ "-a", "--all-users", NULL,
		 N_("consider items from all users"),
		 { 0, 0, 0, 0} },
		{ "", NULL, NULL, NULL, { 0, 0, 0, 0} },
		{ "-h", "--help", NULL,
		 N_("show this help and exit"),
		 { 0, 0, 0, 0} },
		{ "-V", "--version", NULL,
		 N_("show version information and exit"),
		 { 0, 0, 0, 0} },
#ifdef ENABLE_DEBUGGING
		{ "-!", "--debug", N_("FILE"),
		 N_("write debug logs to FILE"),
		 { 0, 0, 0, 0} },
#endif
		/* End of options list. */
		{ "", NULL, NULL, NULL, { 0, 0, 0, 0} },
		/* List of actions. */
		{ "run", NULL, N_("ITEM"), N_("run an item immediately"), { 0, 0, 0, 0} },
		{ "enable", NULL, N_("ITEM"), N_("enable an item so it runs as scheduled"), { 0, 0, 0, 0} },
		{ "disable", NULL, N_("ITEM"), N_("disable an item so it will not run unless forced"), { 0, 0, 0, 0} },
		{ "status", NULL, N_("ITEM"), N_("show an item's current status"), { 0, 0, 0, 0} },
		{ "list", NULL, NULL, N_("list all defined items"), { 0, 0, 0, 0} },
		{ "update", NULL, NULL, N_("update the crontab and the item list file"), { 0, 0, 0, 0} },
		{ "faults", NULL, NULL, N_("list the faults with all defined items"), { 0, 0, 0, 0} },
		/* End of actions list. */
		{ NULL, NULL, NULL, NULL, { 0, 0, 0, 0} }
	};
	unsigned int parameterIndex;
	size_t widestParameterWidth = 0;
	size_t descriptionLeftMargin = 0;
	size_t maxDescriptionWidth = 50;
	size_t rightMargin = 77;
	const char *programDescription;
	const char *bugReportNote;
	unsigned int terminalColumns = 0;

	(void) readTerminalSize(stdout, &terminalColumns, NULL);
	if (terminalColumns > 5) {
		rightMargin = (size_t) (terminalColumns - 3);
	}

	/*@-formatconst@ */
	/*
	 * splint note: unavoidable use of %s in translated string.  Should
	 * be hard to exploit - the message catalogue would have to be
	 * replaced or forced to load from another location.
	 */
	printf(_("Usage: %s [OPTION]... ACTION [ITEM]"), PACKAGE_NAME);
	/*@+formatconst@ */

	printf("\n");

	/*@-mustfreefresh@ */
	/*
	 * splint note: the gettext calls made by _() cause memory leak
	 * warnings, but in this case it's unavoidable, and mitigated by the
	 * fact we only translate each string once.
	 */
	programDescription = _("Run a scheduled command inside a wrapper providing additional features.");
	if (NULL != programDescription) {
		outputWordWrap(stdout, programDescription, rightMargin, 0);
		printf("\n");
	}

	printf("\n");

	/*
	 * Translate the help text, and calculate the displayed width of
	 * each part of each parameter definition.  The total display width
	 * of the short option / action name, long option, and parameter
	 * argument together form the "parameterWidth" - we look for the
	 * widest one to calculate the left margin for all of the
	 * descriptions to start at.
	 */
	for (parameterIndex = 0; NULL != parameterDefinitions[parameterIndex].shortOption; parameterIndex++) {
		struct scwParameterDefinition *definition;
		size_t parameterWidth;

		definition = &(parameterDefinitions[parameterIndex]);
		parameterWidth = 0;

		definition->width.shortOption = calculateDisplayedWidth(definition->shortOption);
		definition->width.longOption = 0;
		definition->width.parameterArgument = 0;
		definition->width.description = 0;

		if (NULL != definition->longOption) {
			definition->width.longOption = calculateDisplayedWidth(definition->longOption);
		}

		if (NULL != definition->parameterArgument) {
			/*@observer@ */ const char *translated;
			translated = _(definition->parameterArgument);
			if (NULL != translated) {
				definition->parameterArgument = translated;
			}
			definition->width.parameterArgument = calculateDisplayedWidth(definition->parameterArgument);
		}

		if (NULL != definition->description) {
			/*@observer@ */ const char *translated;
			translated = _(definition->description);
			if (NULL != translated) {
				definition->description = translated;
			}
			definition->width.description = calculateDisplayedWidth(definition->description);
		}

		/*
		 * The parameterWidth is padded with a left margin of 2
		 * spaces, a ", " between the short and long options, a
		 * space between the long option and the argument, and two
		 * spaces after the argument:
		 *
		 * "  <short>, <long> <arg>  <description>"
		 *
		 * If we don't have getopt_long() then ", <long>" is omitted.
		 */
		parameterWidth += 2 + definition->width.shortOption;	/* "  short" */
#ifdef HAVE_GETOPT_LONG
		if (definition->width.longOption > 0)
			parameterWidth += 2 + definition->width.longOption;	/* ", <long>" */
#endif
		parameterWidth += 1 + definition->width.parameterArgument;	/* " ARG" */
		parameterWidth += 2;	    /* final 2 spaces */

		if (parameterWidth > widestParameterWidth) {
			widestParameterWidth = parameterWidth;
		}
	}

	/*
	 * Set the left margin for the parameter descriptions, based on the
	 * widest parameter width, or (rightMargin - maxDescriptionWidth),
	 * whichever is less, so that there is always room for
	 * "maxDescriptionWidth" characters of description.
	 */
	descriptionLeftMargin = rightMargin - maxDescriptionWidth;
	if (widestParameterWidth < descriptionLeftMargin) {
		descriptionLeftMargin = widestParameterWidth;
	}

	debug("%s: descriptionLeftMargin=%d, widestParameterWidth=%d, rightMargin=%d", "help display",
	      (int) descriptionLeftMargin, (int) widestParameterWidth, (int) rightMargin);

	/*
	 * Display each of the parameter definitions, word wrapping the
	 * descriptions.
	 */
	for (parameterIndex = 0; NULL != parameterDefinitions[parameterIndex].shortOption; parameterIndex++) {
		struct scwParameterDefinition *definition;
		size_t parameterWidth;

		definition = &(parameterDefinitions[parameterIndex]);
		parameterWidth = 0;

		if (definition->width.shortOption > 0 && NULL != definition->shortOption) {
			printf("  %s", definition->shortOption);
			parameterWidth += 2 + definition->width.shortOption;
		}
#ifdef HAVE_GETOPT_LONG
		if (definition->width.longOption > 0 && NULL != definition->longOption) {
			printf(", %s", definition->longOption);
			parameterWidth += 2 + definition->width.longOption;
		}
#endif
		if (definition->width.parameterArgument > 0 && NULL != definition->parameterArgument) {
			printf(" %s", definition->parameterArgument);
			parameterWidth += 1 + definition->width.parameterArgument;
		}

		/* Just start a new line if there's no description. */
		if ((0 == definition->width.description) || (NULL == definition->description)) {
			printf("\n");
			continue;
		}

		/*
		 * If the parameter is too wide, start a new line for the
		 * description.  In both cases, pad with spaces up to the
		 * description left margin.
		 */
		if (parameterWidth >= descriptionLeftMargin) {
			printf("\n%*s", (int) descriptionLeftMargin, "");
		} else if (parameterWidth < descriptionLeftMargin) {
			printf("%*s", (int) (descriptionLeftMargin - parameterWidth), "");
		}

		/* Output the description, word wrapped. */
		outputWordWrap(stdout, definition->description, rightMargin - descriptionLeftMargin,
			       descriptionLeftMargin);

		printf("\n");
	}

	printf("\n");

	bugReportNote = _("Please report any bugs to: %s");
	if (NULL != bugReportNote) {
		/*@-formatconst@ */
		/*
		 * splint note: see earlier "formatconst" note.
		 * flawfinder: same reason.
		 */
		printf(bugReportNote, PACKAGE_BUGREPORT);	/* flawfinder: ignore */
		/*@+formatconst@ */
	}

	printf("\n");
}
