/*
 * Functions for the "list" action.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <string.h>
#include <sys/file.h>

/*
 * List all defined items.
 */
int showItemList(struct scwState *state)
{
	char **usernameArray = NULL;
	size_t usernameCount = 0;
	size_t usernameIndex;

	if (state->allUsers) {
		enumerateAllUsers(state, &usernameArray, &usernameCount);
	} else {
		usernameArray = (char **) stringCopy(state, NULL, sizeof(char *) - 1);
		if (NULL == usernameArray)
			return SCW_EXIT_ERROR;
		/*@-dependenttrans@ */
		usernameArray[0] = state->username;
		/*@+dependenttrans@ */
		usernameCount = 1;
	}

	for (usernameIndex = 0; NULL != usernameArray && usernameIndex < usernameCount; usernameIndex++) {
		char *originalUsername;
		size_t originalUsernameLength;
		char *username;
		size_t usernameLength;
		char **itemArray = NULL;
		size_t itemCount = 0;
		size_t itemIndex;

		username = usernameArray[usernameIndex];
		if (NULL == username)
			continue;
		usernameLength = strlen(username);	/* flawfinder: ignore */
		/*
		 * flawfinder warns of strlen() with non-null-terminated
		 * strings; the array contains strings we have explicitly
		 * null terminated.
		 */

		originalUsername = state->username;
		originalUsernameLength = state->usernameLength;
		state->username = username;
		state->usernameLength = usernameLength;

		/* Load the per-user configuration for this user. */
		memset(&(state->userSettings), 0, sizeof(state->userSettings));
		if (0 != loadUserConfig(state)) {
			state->username = originalUsername;
			state->usernameLength = originalUsernameLength;
			continue;
		}

		/* Find the items for this user. */
		enumerateUserItems(state, &itemArray, &itemCount);

		/* Load each item and display its information. */
		for (itemIndex = 0; NULL != itemArray && itemIndex < itemCount; itemIndex++) {
			struct scwSettings combinedItemSettings;
			const char *metricsDir;
			size_t metricsDirLength;
			bool itemDisabled = false;

			state->item = itemArray[itemIndex];
			if (NULL == state->item)
				continue;
			state->itemLength = strlen(state->item);	/* flawfinder: ignore */
			/*
			 * flawfinder warns of strlen() with
			 * non-null-terminated strings; the array contains
			 * strings we have explicitly null terminated.
			 */

			/* Load the item's settings. */
			memset(&(state->itemSettings), 0, sizeof(state->itemSettings));
			if (0 != loadCurrentItemSettings(state))
				continue;

			/* Combine and expand the settings for this item. */
			memset(&combinedItemSettings, 0, sizeof(combinedItemSettings));
			if (0 != combineSettings(state, &combinedItemSettings))
				continue;
			expandAllRawValues(state, &combinedItemSettings);

			metricsDir = combinedItemSettings.metricsDir.expandedValue;
			metricsDirLength = combinedItemSettings.metricsDir.expandedLength;

			/* Check whether the item is disabled. */
			if (NULL != metricsDir) {
				if (fileExistsAt(metricsDir, metricsDirLength, STATIC_STRING("disabled"))) {
					itemDisabled = true;
				}
			}

			/* Skip if disabled and we're only listing enabled items. */
			if (state->listEnabledOnly && itemDisabled)
				continue;

			/* Skip if enabled and we're only listing disabled items. */
			if (state->listDisabledOnly && (!itemDisabled))
				continue;

			/* Display the information. */
			/*@-mustfreefresh@ */
			/* splint note: gettext triggers a warning we can't resolve. */
			printf("%.*s\t%.*s\t%s\t", (int) (state->usernameLength), state->username,
			       (int) (state->itemLength), state->item, itemDisabled ? _("disabled") : _("enabled"));
			if (NULL == combinedItemSettings.description.expandedValue
			    || combinedItemSettings.description.expandedLength < 1) {
				printf("%s", _("(no description)"));
			} else {
				printf("%.*s", (int) (combinedItemSettings.description.expandedLength),
				       combinedItemSettings.description.expandedValue);
			}
			/*@+mustfreefresh@ */
			printf("\n");

			/* Display extra information, if the option was selected. */
			/*@-mustfreefresh@ */
			/* splint note: gettext triggers warnings we can't resolve. */
			if (state->listWithInfo) {
				size_t arrayIndex;
				int checkLockDescriptor;
				struct scwItemStatus itemStatus;

				/* Show the item's schedule. */
				for (arrayIndex = 0; arrayIndex < combinedItemSettings.countSchedules; arrayIndex++) {
					if (NULL == combinedItemSettings.schedule[arrayIndex].expandedValue
					    && combinedItemSettings.schedule[arrayIndex].rawValue != NULL)
						expandRawValue(state, &(combinedItemSettings.schedule[arrayIndex]));
					if (NULL != combinedItemSettings.schedule[arrayIndex].expandedValue)
						printf(" -- %s: %.*s\n", _("Schedule"),
						       (int) (combinedItemSettings.schedule[arrayIndex].expandedLength),
						       combinedItemSettings.schedule[arrayIndex].expandedValue);
				}

				/* Show the item's success interval. */
				if (combinedItemSettings.numSuccessInterval > 0) {
					printf(" -- %s: %s\n", _("Success interval"),
					       timePeriodString((int) (combinedItemSettings.numSuccessInterval)));
				}

				/* Attempt to lock the CheckLockFile. */
				if (NULL != combinedItemSettings.checkLockFile.expandedValue) {
					checkLockDescriptor =
					    fileOpenForAppend(combinedItemSettings.checkLockFile.expandedValue);
					if (checkLockDescriptor >= 0) {
						if (0 != flock(checkLockDescriptor, LOCK_EX)) {
							(void) close(checkLockDescriptor);
							checkLockDescriptor = -1;
						}
					}
				} else {
					checkLockDescriptor = -1;
				}

				/* Load the item's metrics. */
				memset(&itemStatus, 0, sizeof(itemStatus));
				(void) loadStatusFromMetrics(&(combinedItemSettings.metricsDir), &itemStatus,
							     checkLockDescriptor < 0 ? true : false);

				/* Release the lock. */
				if (checkLockDescriptor >= 0) {
					(void) flock(checkLockDescriptor, LOCK_UN);
					(void) close(checkLockDescriptor);
				}

				if (itemStatus.tooLongSinceLastSuccess)
					printf(" !! %s\n", _("This item has gone too long without a successful run."));
				if (itemStatus.isOverrunning)
					printf(" !! %s\n", _("This item is currently overrunning."));

				printf(" -- %s %s\n", _("Last end time:"), timeAndDateString(itemStatus.lastEnded));
				if (itemStatus.lastRunTime > 0)
					printf(" -- %s %s\n", _("Time elapsed during last run:"),
					       timePeriodString((int) (itemStatus.lastRunTime)));

				if (itemStatus.isRunning)
					printf(" -- %s %s\n", _("Running since"),
					       timeAndDateString(itemStatus.lastStarted));

				printf("\n");
			}
		}
		/*@+mustfreefresh@ */

		state->item = NULL;
		state->itemLength = 0;

		state->username = originalUsername;
		state->usernameLength = originalUsernameLength;
	}

	return 0;
}
