/*
 * Main program entry point - read the command line options, then perform
 * the appropriate actions.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>
#include <ctype.h>
#include <sys/utsname.h>
#include <time.h>


/*
 * Get the system hostname, add it to the
 * strings pool, and point state->hostname to it.  Populate
 * state->hostnameLength when doing so.
 */
static void getHostname(struct scwState *state)
{
	struct utsname unameInfo;
	size_t hostnameLength;
	char *hostnameCopy;

	memset(&unameInfo, 0, sizeof(unameInfo));
	(void) uname(&unameInfo);

	hostnameLength = strlen(unameInfo.nodename);	/* flawfinder: ignore */
	/*
	 * flawfinder rationale: the objection to strlen() is that it does
	 * not handle strings that are not null-terminated - but the fields
	 * that uname() returns are null-terminated, so we ignore the
	 * warning.
	 */
	hostnameCopy = stringCopy(state, unameInfo.nodename, hostnameLength);

	if (NULL != hostnameCopy) {
		state->hostname = hostnameCopy;
	} else {
		debug("%s", "stringCopy");
		/*@-observertrans@ */
		state->hostname = "unknown";
		/*@+observertrans@ */
		hostnameLength = strlen(state->hostname);	/* flawfinder: ignore */
		/* flawfinder - strlen() again, and this is definitely null-terminated. */
	}

	state->hostnameLength = hostnameLength;

	debugShowStringValue("hostname", state->hostname);
}


/*
 * Get the username of the account we are running under, add it to the
 * strings pool, and point state->username to it.  Populate
 * state->usernameLength when doing so.
 */
static void getUsername(struct scwState *state)
{
	uid_t uid;
	struct passwd *passwdEntry;
	char *usernameString;
	size_t usernameLength;
	char *usernameCopy;

	usernameString = "unknown";

	/*@-type@ */
	/* splint sees __uid_t as different to uid_t. */
	uid = getuid();
	passwdEntry = getpwuid(uid);
	/*@+type@ */

	/*@-branchstate@ */
	/*
	 * splint warns that usernameString can be end up either
	 * static or local here, but it doesn't matter because we're
	 * just going to use it once immediately, it's never
	 * referenced again.
	 */
	if (NULL != passwdEntry) {
		usernameString = passwdEntry->pw_name;
	} else {
		debug("%s: %s", "getpwuid failed", strerror(errno));
		if (0 == uid) {
			usernameString = "root";
		}
	}
	/*@+branchstate@ */

	usernameLength = strlen(usernameString);	/* flawfinder: ignore */
	/*
	 * flawfinder rationale: the objection to strlen() is that
	 * it does not handle strings that are not null-terminated -
	 * but the fields that getpwuid() returns are
	 * null-terminated, so we ignore the warning.
	 */
	usernameCopy = stringCopy(state, usernameString, usernameLength);

	if (NULL != usernameCopy) {
		/*
		 * Originally we were going to normalise the username to
		 * lower case, in an attempt to simplify administration; but
		 * turning a username back to a UID won't work with a
		 * case-sensitive username, so we treat usernames as
		 * case-sensitive.
		 */
		/*
		   size_t charIndex;
		   for (charIndex = 0; charIndex < usernameLength; charIndex++) {
		   usernameCopy[charIndex] = tolower(usernameCopy[charIndex]);
		   }
		 */
		state->username = usernameCopy;
	} else {
		debug("%s", "stringCopy");
		/*@-observertrans@ */
		state->username = "unknown";
		/*@+observertrans@ */
		usernameLength = strlen(state->username);	/* flawfinder: ignore */
		/* flawfinder - strlen() again, and this is definitely null-terminated. */
	}

	state->usernameLength = usernameLength;

	debugShowStringValue("username", state->username);
}


/*
 * Get today's date in YYYY-MM-DD format, add it to the strings pool, and
 * point state->currentDate to it.  Populate state->currentDateLength when
 * doing so.
 */
static void getDate(struct scwState *state)
{
	time_t epochTime;
	struct tm *brokenDownTime;
	char timestampBuffer[16];	 /* flawfinder: ignore */
	char *currentDateString;
	size_t currentDateLength;

	/*
	 * flawfinder notes: timestampBuffer is only written to by
	 * strftime(), which takes a buffer size, and we enforce string
	 * termination.
	 */

	epochTime = time(NULL);
	timestampBuffer[0] = '\0';
	brokenDownTime = localtime(&epochTime);
	if (0 == strftime(timestampBuffer, sizeof(timestampBuffer), "%Y-%m-%d", brokenDownTime)) {
		timestampBuffer[0] = '\0';
	}
	timestampBuffer[sizeof(timestampBuffer) - 1] = '\0';	/* enforce termination */
	currentDateLength = strlen(timestampBuffer);	/* flawfinder: ignore */
	/* flawfinder - strlen() has been given a null-terminated string. */

	currentDateString = stringCopy(state, timestampBuffer, currentDateLength);

	if (NULL != currentDateString) {
		state->currentDate = currentDateString;
		state->currentDateLength = currentDateLength;
	}

	debugShowStringValue("currentDate", state->currentDate);
}


/*
 * Load the global configuration, returning nonzero on error.
 */
static int loadGlobalConfig(struct scwState *state)
{
	const char *globalConfigFile;

	globalConfigFile = state->configFile;
	if (NULL == globalConfigFile)
		globalConfigFile = SYSCONFDIR "/" PACKAGE_NAME "/default.cf";

	return loadSettings(state, globalConfigFile, &(state->globalSettings), SCW_SOURCE_GLOBAL_CONFIG);
}


/*
 * Load the per-user configuration for the current user, if there is any,
 * returning nonzero on error.
 */
int loadUserConfig(struct scwState *state)
{
	const char *userConfigFile;

	/*
	 * Per-user configuration.  Note that a command-line "--set" option
	 * could have changed the user config file location.
	 */
	if (NULL != state->commandLineSettings.userConfigFile.rawValue) {
		expandRawValue(state, &(state->commandLineSettings.userConfigFile));
	}
	userConfigFile = state->commandLineSettings.userConfigFile.expandedValue;
	if (NULL == userConfigFile) {
		expandRawValue(state, &(state->globalSettings.userConfigFile));
		userConfigFile = state->globalSettings.userConfigFile.expandedValue;
	}

	return loadSettings(state, userConfigFile, &(state->userSettings), SCW_SOURCE_USER_CONFIG);
}


/*
 * Load the global configuration, then the per-user configuration, returning
 * non-zero on error.
 */
static int loadConfigFiles(struct scwState *state)
{
	int retcode;

	retcode = loadGlobalConfig(state);
	if (0 != retcode)
		return retcode;
	return loadUserConfig(state);
}


/*
 * Process command-line arguments and set option flags, then call functions
 * to initialise, and finally enter the main loop.
 */
int main(int argc, char **argv)
{
	struct scwState *state;
	int retcode = SCW_EXIT_SUCCESS;

	state = newState();
	if (NULL == state)
		return SCW_EXIT_ERROR;

#ifdef ENABLE_NLS
	/* Initialise language translation. */
	(void) setlocale(LC_ALL, "");
	(void) bindtextdomain(PACKAGE, LOCALEDIR);
	(void) textdomain(PACKAGE);
#endif

	/* Parse the command line arguments. */
	retcode = parseOptions(state, argc, argv);
	if (0 != retcode) {
		debug("%s: %d", "exiting with status", retcode);
		clearState(state);
		free(state);
		return retcode;
	}

	/* Determine the current hostname. */
	getHostname(state);

	/* Determine the current username. */
	getUsername(state);

	/* Determine today's date. */
	getDate(state);

	/* Set default values. */
	setGlobalDefaults(state);

	/* Parse the configuration files. */
	if (SCW_ACTION_NONE != state->action && SCW_ACTION_HELP != state->action && SCW_ACTION_VERSION != state->action) {
		retcode = loadConfigFiles(state);
		if (0 != retcode) {
			debug("%s: %d", "exiting with status", retcode);
			clearState(state);
			free(state);
			return retcode;
		}
	}

	/* Parse the settings for the selected item. */
	if (NULL != state->item) {
		retcode = loadCurrentItemSettings(state);
		if (0 != retcode) {
			debug("%s: %d", "exiting with status", retcode);
			clearState(state);
			free(state);
			return retcode;
		}
	}

	/* If there is no output map, set a default one. */
	setDefaultOutputMap(state);

#ifdef ENABLE_DEBUGGING
	debugOutputAllSettings("globalSettings", &(state->globalSettings));
	debugOutputAllSettings("userSettings", &(state->userSettings));
	debugOutputAllSettings("itemSettings", &(state->itemSettings));
	debugOutputAllSettings("commandLineSettings", &(state->commandLineSettings));
#endif

	/* Check whether we're running on a terminal. */
	state->runningOnTerminal = false;
	if ((1 == isatty(STDIN_FILENO)) && (1 == isatty(STDOUT_FILENO)) && (1 == isatty(STDERR_FILENO)))
		state->runningOnTerminal = true;

	/*
	 * Make fd 3 a duplicate of fd 2, leaving it available to be
	 * repurposed for a status stream in child processes.
	 */
	(void) dup2(2, 3);

	/* Run the appropriate action. */
	switch (state->action) {
	case SCW_ACTION_NONE:
		break;
	case SCW_ACTION_HELP:
		displayHelp();
		break;
	case SCW_ACTION_VERSION:
		displayVersion();
		break;
	case SCW_ACTION_RUN:
		retcode = runItem(state);
		break;
	case SCW_ACTION_ENABLE:
		retcode = setItemEnabledState(state, true);
		break;
	case SCW_ACTION_DISABLE:
		retcode = setItemEnabledState(state, false);
		break;
	case SCW_ACTION_STATUS:
		retcode = showItemStatus(state);
		break;
	case SCW_ACTION_LIST:
		retcode = showItemList(state);
		break;
	case SCW_ACTION_UPDATE:
		retcode = updateGlobalFiles(state);
		break;
	case SCW_ACTION_FAULTS:
		retcode = listFaults(state);
		break;
	}

	debug("%s: %d", "exiting with status", retcode);

	/* Close the fd 3 we opened earlier. */
	(void) close(3);

	clearState(state);
	free(state);

	debugWriteOutput("", "", -1, "");   /* close debug stream. */

	return retcode;
}
