/*
 * Functions internal to program.  Include "config.h" first.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#ifndef _SCW_INTERNAL_H
#define _SCW_INTERNAL_H 1

#include "config.h"
#include <stdio.h>
#include <stddef.h>
#include <unistd.h>
#include <sys/stat.h>

/*
 * Limits.
 */
#define SCW_MAX_VALUES		16	/* arrays of values for settings */
#define SCW_MAX_LINELENGTH	1024	/* settings and spool copying */
#define SCW_MAX_READBUFFER	131072	/* reading from spawned command */
#define SCW_MAX_RELAYBUFFER	16384	/* relaying pipe to socket */
#define SCW_MAX_JSONSTRING	4096	/* constructing JSON strings */

/*
 * Exit status constants.
 */
/* General exit status values. */
#define SCW_EXIT_SUCCESS            0
#define SCW_EXIT_BAD_ARGS           5
#define SCW_EXIT_BAD_CONFIG         6
#define SCW_EXIT_ERROR              7
/* Exit status values for "run". */
#define SCW_EXIT_RUN_SUCCEEDED      0
#define SCW_EXIT_RUN_FAILED         1
#define SCW_EXIT_RUN_TIMEOUT        2
#define SCW_EXIT_RUN_NOCHECKS_OK    3
#define SCW_EXIT_RUN_NOCHECKS_FAIL  4
#define SCW_EXIT_RUN_COMMAND        8
#define SCW_EXIT_RUN_DISABLED       9
#define SCW_EXIT_RUN_PREREQUISITE   10
#define SCW_EXIT_RUN_DEPENDENCY     11
#define SCW_EXIT_RUN_CONFLICT       12
#define SCW_EXIT_RUN_CONCURRENCY    13
#define SCW_EXIT_RUN_MININTERVAL    14
/* Exit status values for "status". */
#define SCW_EXIT_STATUS_NONEXISTENT 16
#define SCW_EXIT_STATUS_DISABLED    32
#define SCW_EXIT_STATUS_RUNNING     64

/*
 * Switches.
 */
/* Whether to close stdin when running an item's Command. */
#define CLOSE_COMMAND_STDIN 0	/* No - so cron's "%" syntax can work. */


/*
 * Strategies for receiving and recording command output.
 */
typedef enum {
	SCW_RECEIVER_PIPE,		/* command writes to parent over pipes */
	SCW_RECEIVER_UNIXSOCKET,	/* command sends to Unix socket */
	SCW_RECEIVER_RELAY,		/* relay to Unix socket via pipes to subprocesses */
	SCW_RECEIVER_AUTO		/* automatically choose one of the above */
} scwReceiverStrategy;

/*
 * A pointer to a char *, whose value is allowed to be NULL.
 */
typedef /*@null@ */ char *nullable_string_t;

/*
 * The value of a setting, which consists of a raw value (as it was taken
 * from the configuration or command line), and an expanded value (after
 * placeholders have been replaced).  The rawValue will be NULL unless the
 * setting exists.  The expandedValue will be NULL until the rawValue is
 * expanded.  Neither should be passed to free().
 *
 * The rawValue string may not be null-terminated but the expandedValue
 * string will be.  Both of their lengths are held here to keep it
 * consistent and avoid the need for repeated strlen() calls elsewhere.
 */
struct scwSettingValue {
	/*@null@*/ /*@dependent@*/ const char *rawValue;
	/*@null@*/ /*@dependent@*/ char *expandedValue;
	size_t rawLength;
	size_t expandedLength;
};

/*
 * All possible settings that can be read from a file.
 */
struct scwSettings {
	/* Global-only settings. */
	struct scwSettingValue userConfigFile;
	struct scwSettingValue itemListFile;
	struct scwSettingValue crontabFile;
	struct scwSettingValue updateLockFile;
	/* Global and per-user settings. */
	struct scwSettingValue itemsDir;
	struct scwSettingValue metricsDir;
	struct scwSettingValue checkLockFile;
	struct scwSettingValue sendmail;
	struct scwSettingValue transmitForm;
	struct scwSettingValue transmitJson;
	/* Item settings. */
	struct scwSettingValue description;
	struct scwSettingValue command;
	struct scwSettingValue ambiguousExitStatus;
	struct scwSettingValue schedule[SCW_MAX_VALUES];
	struct scwSettingValue randomDelay;
	struct scwSettingValue maxRunTime;
	struct scwSettingValue prerequisite;
	struct scwSettingValue minInterval;
	struct scwSettingValue successInterval;
	struct scwSettingValue concurrencyWait;
	struct scwSettingValue silentConcurrency;
	struct scwSettingValue ignoreOverrun;
	struct scwSettingValue dependsOn[SCW_MAX_VALUES];
	struct scwSettingValue dependencyWait;
	struct scwSettingValue silentDependency;
	struct scwSettingValue conflictsWith[SCW_MAX_VALUES];
	struct scwSettingValue conflictWait;
	struct scwSettingValue silentConflict;
	struct scwSettingValue statusMode;
	struct scwSettingValue statusTag;
	struct scwSettingValue timestampUTC;
	struct scwSettingValue httpInterval;
	struct scwSettingValue httpTimeout;
	struct scwSettingValue sharedSecret;
	struct scwSettingValue emailMaxBodySize;
	struct scwSettingValue emailBodyText;
	struct scwSettingValue emailAttachmentName;
	struct scwSettingValue emailSender;
	struct scwSettingValue emailSubject;
	struct scwSettingValue outputMap[SCW_MAX_VALUES];
	struct scwSettingValue receiverStrategy;
	/* Value counts for settings with multiple values. */
	size_t countSchedules;
	size_t countDependencies;
	size_t countConflicts;
	size_t countOutputMaps;
	/* Numeric values. */
	unsigned int numAmbiguousExitStatus;
	unsigned int numRandomDelay;
	unsigned int numMaxRunTime;
	unsigned int numMinInterval;
	unsigned int numSuccessInterval;
	unsigned int numConcurrencyWait;
	unsigned int numDependencyWait;
	unsigned int numConflictWait;
	unsigned int numHTTPInterval;
	unsigned int numHTTPTimeout;
	unsigned int numEmailMaxBodySize;
	/* Enumerated values. */
	scwReceiverStrategy enumReceiverStrategy;
	/* Boolean values. */
	bool flagSilentConcurrency;
	bool flagIgnoreOverrun;
	bool flagSilentDependency;
	bool flagSilentConflict;
	bool flagTimestampUTC;
	/* Flags to say whether multi-value settings were cleared. */
	bool clearedSchedules;
	bool clearedDependencies;
	bool clearedConflicts;
	bool clearedOutputMaps;
};

/*
 * Sources for a setting.
 */
typedef enum {
	SCW_SOURCE_GLOBAL_CONFIG,
	SCW_SOURCE_USER_CONFIG,
	SCW_SOURCE_ITEM_SETTINGS,
	SCW_SOURCE_ITEM_SCRIPT,
	SCW_SOURCE_COMMANDLINE
} scwSettingSource;

/*
 * Format types for an output map.
 */
typedef enum {
	SCW_FORMAT_RAW,
	SCW_FORMAT_STAMPED,
	SCW_FORMAT_JSON,
	SCW_FORMAT_FORM
} scwOutputFormat;

/*
 * Destination types for an output map.
 */
typedef enum {
	SCW_DESTINATION_FILE,
	SCW_DESTINATION_SYSLOG,
	SCW_DESTINATION_EMAIL,
	SCW_DESTINATION_URL
} scwOutputDestinationType;

/*
 * Output stream identifiers.  These can form a bitmap.
 */
typedef enum {
	SCW_STREAM_STDOUT = 1,
	SCW_STREAM_STDERR = 2,
	SCW_STREAM_STATUS = 4
} scwOutputStreamId;

/*
 * Structure defining a parsed output map item.  The destination and (if
 * set) the temporarySpool both point inside the scwState strings array, so
 * none of these pointers should be passed to free().
 */
struct scwOutput {
	 /*@dependent@*/ const char *destination;
	 /*@dependent@*/ /*@null@*/ const char *temporarySpool;
	scwOutputFormat format;
	scwOutputDestinationType destinationType;
	int syslogPriority;	/* syslog(3) facility and priority */
	int streamsIncluded;	/* sum of scwOutputStreamId values */
	int writeDescriptor;	/* open file descriptor to write to */
	unsigned int lineNumber; /* output line number being written */
	time_t spoolUntil;	/* when to next send by HTTP/HTTPS */
	time_t nextFileCheck;	/* when to next check the output file */
	ino_t destinationInode; /* last known output inode, for ">" */
	bool sendOnFailure;	/* "!" mode - spool, send on failure */
	bool outputFileCheck;	/* ">" mode - re-open file if rotated */
	bool anythingWritten;	/* true if anything has been written */
};

/*
 * Structure holding information about the process executing the item's
 * command.  Any of the file descriptors may be -1, if not in use.
 */
struct scwCommandProcess {
	/*@dependent@*/ struct scwSettings *settings;	/* settings applicable to the item */
	/*@dependent@*/ /*@null@*/ char *workDir;	/* temporary working directory, if any */
	pid_t pid;		/* process ID */
	time_t startTime;	/* epoch time it started */
	int stdoutDescriptor;	/* file descriptor to read command stdout from */
	int stderrDescriptor;	/* file descriptor to read command stderr from */
	int statusDescriptor;	/* file descriptor to read command status from */
	int combinedDescriptor;	/* file descriptor to read all streams from */
	int itemLockDescriptor;	/* file descriptor holding the item lock */
	int exitStatus;		/* the command's exit status, if it has ended */
	scwReceiverStrategy strategy;   /* which receiver strategy is being used */
	bool closeInput;	/* whether to close the command's stdin */
	bool ended;		/* whether the command has ended yet */
	bool timedOut;		/* whether the command reached its MaxRunTime */
};

/*
 * Available actions.
 */
typedef enum {
	SCW_ACTION_NONE,
	SCW_ACTION_HELP,
	SCW_ACTION_VERSION,
	SCW_ACTION_RUN,
	SCW_ACTION_ENABLE,
	SCW_ACTION_DISABLE,
	SCW_ACTION_STATUS,
	SCW_ACTION_LIST,
	SCW_ACTION_UPDATE,
	SCW_ACTION_FAULTS
} scwAction;

/*
 * Structure holding global state.
 */
struct scwState {
	/* Settings. */
	struct scwSettings globalSettings;	/* settings from global config */
	struct scwSettings userSettings;	/* settings from per-user config */
	struct scwSettings itemSettings;	/* settings from current item */
	struct scwSettings commandLineSettings;	/* settings from command line */

	/* Parsed details for the current item. */
	struct scwOutput output[SCW_MAX_VALUES];	/* active outputs */
	size_t outputCount;
	/*@null@*/ /*@dependent@*/ const char *itemCommand;
	size_t itemCommandLength;

	/* Value string allocation pool. */
	/*@null@*/ /*@keep@*/ char **strings;	 /* array of allocated strings */
	size_t stringsArraySize;		 /* length allocated to array */
	size_t countStrings;			 /* number of strings allocated */

	/* Parameters and flags from the command line and environment. */
	/*@null@*/ /*@dependent@*/ char *hostname;	/* system hostname */
	size_t hostnameLength;
	/*@null@*/ /*@dependent@*/ char *username;	/* current username */
	size_t usernameLength;
	/*@null@*/ /*@dependent@*/ char *currentDate;	/* today's date */
	size_t currentDateLength;
	/*@null@*/ /*@dependent@*/ const char *configFile;	/* global config file */
	/*@null@*/ /*@dependent@*/ const char *item;	/* currently selected item */
	size_t itemLength;
	/*@null@*/ /*@keep@*/ char **arguments;		/* remaining arguments */
	size_t argumentCount;
	scwAction action;			/* which action to perform */
	scwOutputStreamId streamForStatus;	/* which stream to read status from */
	bool forceRun;				/* run if disabled (--force) */
	bool listEnabledOnly;			/* only list enabled (--enabled) */
	bool listDisabledOnly;			/* only list disabled (--enabled) */
	bool listWithInfo;			/* detailed list (--info) */
	bool allUsers;				/* items of all users (--all-users) */
	bool itemHasSettingsFile;		/* item settings file available */
	bool runningOnTerminal;			/* produce colourised tty output */
	bool timestampUTC;			/* express timestamps in UTC */
	bool continueWithoutCheckLock;		/* continue if CheckLockFile not writable */
	bool continueWithoutItemLock;		/* continue if item lock file not writable */
	bool continueWithoutMetrics;		/* continue if MetricsDir is inaccessible */
	bool continueWithoutOutputFile;		/* continue if an OutputMap file is inaccessible */
};

/*
 * Structure holding the status of an item, derived from its metrics files.
 */
struct scwItemStatus {
	time_t whenDisabled;
	time_t whenPrerequisitesLastMet;
	time_t lastStarted;
	time_t lastEnded;
	time_t lastSucceeded;
	time_t firstFailed;
	time_t firstOverran;
	time_t lastStatusWritten;
	time_t whenDelayChosen;
	unsigned int successInterval;
	unsigned int currentDelay;
	unsigned int lastRunTime;
	pid_t pid;
	char lastStatus[1024];
	/*
	 * Flags derived from the information above.
	 */
	bool isDisabled;
	bool hasPrerequisitesMet;
	bool hasNeverRun;
	bool isRunning;
	bool isOverrunning;
	bool tooLongSinceLastSuccess;
	bool lastRunFailed;
	bool willRunAfterDelay;
};

/* Macro for passing static strings and their length together. */
#define STATIC_STRING(x) x, strlen(x)	    /* flawfinder: ignore */
/*
 * flawfinder - calling strlen() on static strings guaranteed to be
 * null-terminated is OK.
 */

/* Macro for getting the length of a static string. */
#define STATIC_STRLEN(x) strlen(x)	    /* flawfinder: ignore */
/* flawfinder - same rationale as above. */

/*
 * State allocation and initialisation functions.
 */

/* Allocate a new, empty, state structure. */
/*@null@*/ struct scwState * newState(void);

/* Free any allocated memory in the state, and clear it completely. */
void clearState(/*@null@*/ struct scwState *);

/* Set the global default settings. */
void setGlobalDefaults(struct scwState *);

/* Set a default output map if none has been defined. */
void setDefaultOutputMap(struct scwState *);

/* Parse command line options and populate the initial state. */
int parseOptions(struct scwState *, int, char **);

/*
 * Functions related to word wrapping.
 */

/* Read the size of the terminal. */
bool readTerminalSize(FILE *, /*@null@*/ unsigned int *, /*@null@*/ unsigned int *);

/* Determine how many display columns a string will occupy. */
size_t calculateDisplayedWidth(const char *);

/* Output a string with word wrapping and a left margin. */
void outputWordWrap(FILE *, const char *, size_t, size_t);

/*
 * Functions for the main program actions.
 */

/* Display the command-line help. */
void displayHelp(void);

/* Display the version information. */
void displayVersion(void);

/* Run the selected scheduled command. */
int runItem(struct scwState *);

/* Set the selected item to be either enabled or disabled. */
int setItemEnabledState(struct scwState *, bool);

/* Read the status of an item from its metrics directory. */
bool loadStatusFromMetrics(struct scwSettingValue *, struct scwItemStatus *, bool);

/* Show current status of the selected item. */
int showItemStatus(struct scwState *);

/* List all defined items. */
int showItemList(struct scwState *);

/* Update the crontab and the item list file. */
int updateGlobalFiles(struct scwState *);

/* List the faults with all items. */
int listFaults(struct scwState *);

/*
 * Miscellaneous functions.
 */

/* Determine whether an item is currently running. */
bool itemIsRunning(struct scwSettingValue *);

/* Enumerate users by looking for user config and items. */
void enumerateAllUsers(struct scwState *, char ***, size_t *);

/* Enumerate a user's items. */
void enumerateUserItems(struct scwState *, char ***, size_t *);

/*
 * String allocation and formatting functions.
 */

/* Allocate space for a string, optionally duplicating an original. */
/*@null@*/ /*@dependent@*/ char *stringCopy(struct scwState *, /*@null@*/ const char *, size_t);

/* Append a string to another string in a buffer of fixed size. */
size_t stringAppend(size_t, char *, size_t, const char *, size_t);

/* Test whether the first string starts with the second. */
bool stringStartsWith(const char *, const char *);

/* Return a static buffer containing a timestamp string. */
char *timeAndDateString(time_t);

/* Return a static buffer containing a time period expressed as a string. */
char *timePeriodString(int);

/* Populate a buffer with a copy of a string, escaped for JSON. */
/*@dependent@ */ char *jsonEscapeString(/*@dependent@ */ char *, size_t, /*@null@ */ const char *, size_t, /*@null@ */ size_t *);

/* Populate a buffer with a copy of a string, URL-encoded. */
/*@dependent@ */ char *urlEncodeString(/*@dependent@ */ char *, size_t, /*@null@ */ const char *, size_t, /*@null@ */ size_t *);

/*
 * Functions for handling settings.
 */

/* Load the per-user configuration for the current user. */
int loadUserConfig(struct scwState *);

#ifdef ENABLE_DEBUGGING
/* Debugging output showing all settings. */
void debugOutputAllSettings(const char *, struct scwSettings *);
#endif

/* Output all settings to stdout. */
void outputAllSettings(struct scwState *, struct scwSettings *);

/* Return true if the string is a valid item name. */
bool validateItemName(const char *, size_t);

/* Populate the expandedValue of a setting by expanding placeholders in its rawValue. */
void expandRawValue(struct scwState *, struct scwSettingValue *);

/* Copy a value and re-expand it for another item. */
void expandValueForOtherItem(struct scwState *, /*@null@*/ struct scwSettingValue *, /*@dependent@*/ /*@null@*/ const char *, size_t, struct scwSettingValue *);

/* Populate the expandedValue of all string settings. */
void expandAllRawValues(struct scwState *, struct scwSettings *);

/* Parse a single setting=value string into a settings structure. */
int parseSetting(/*@dependent@*/ const char *, size_t, /*@dependent@*/ struct scwSettings *, scwSettingSource, /*@null@*/ const char *, size_t);

/* Load settings from a file into a settings structure. */
int loadSettings(struct scwState *, /*@null@*/ const char *, /*@dependent@*/ struct scwSettings *, scwSettingSource);

/* Load settings for an item into the item settings structure. */
int loadCurrentItemSettings(struct scwState *);

/* Combine all loaded settings into one structure. */
int combineSettings(struct scwState *, struct scwSettings *);

/*
 * Functions related to recording the output of an item.
 */

/* Capture the command's output using pipes to the main process. */
int captureCommandOutputViaPipe(struct scwState *, struct scwCommandProcess *);

/* Capture the command's output using Unix sockets. */
int captureCommandOutputViaUnixSocket(struct scwState *, struct scwCommandProcess *);

/* Parse a single OutputMap. */
int parseOutputMap(/*@null@*/ struct scwState *, const char *, size_t, /*@null@*/ struct scwOutput *);

/* Populate state->output[] from all OutputMaps in a settings structure. */
int parseAllOutputMaps(struct scwState *, struct scwSettings *);

/* Open all item output streams. */
int openOutputStreams(struct scwState *);

/* Close all open item output file streams. */
void closeOutputStreams(struct scwState *);

/* Send all spooled / deferred output (such as email). */
void sendSpooledOutput(struct scwState *, struct scwSettings *, pid_t, bool);

/* Transmit to URLs according to the HTTPInterval. */
void transmitDelayedUrlSpools(struct scwState *, struct scwSettings *);

/* Remove all temporary files for spooled output. */
void removeSpooledOutput(struct scwState *);

/* Record a line of item output to the appropriate places. */
void recordOutput(struct scwState *, struct scwSettings *, pid_t, scwOutputStreamId, const char *, size_t);

/*
 * Functions for files and directories.
 */

/* Open a file for reading. */
int fileOpenForRead(const char *);

/* Open a file stream for reading. */
/*@null@*/ /*@dependent@*/ FILE *fileOpenStreamForRead(const char *);

/* Create a directory. */
int directoryCreate(const char *, size_t);

/* Create the parent directory of a path. */
int directoryParentCreate(const char *, size_t);

/* Create a file under a directory. */
int fileCreateAt(const char *, size_t, const char *, size_t, bool);

/* Delete a file under a directory. */
int fileDeleteAt(const char *, size_t, const char *, size_t);

/* Open a file for appending to. */
int fileOpenForAppend(const char *);

/* Open a file for appending to, under a directory. */
int fileOpenForAppendAt(const char *, size_t, const char *, size_t);

/* Replace a file's contents atomically, under a directory. */
int fileReplaceContentsAt(const char *, size_t, const char *, size_t, const char *, ...);

/* Return true if a file exists under a directory. */
bool fileExistsAt(const char *, size_t, const char *, size_t);

/* Run stat() on a file under a directory. */
int fileStatAt(const char *, size_t, const char *, size_t, struct stat *);

/* Open a file under a directory, for reading. */
/*@null@*/ /*@dependent@*/ FILE *fileOpenStreamForReadAt(const char *, size_t, const char *, size_t);

/* Open a file under a directory, and read an integer from it. */
int fileReadIntegerFromFileAt(const char *, size_t, const char *, size_t);

/* Write a complete buffer to a file descriptor. */
bool fileWriteBuffer(int, const char *, size_t);

/* Create a temporary file. */
bool tempFileCreate(struct scwState *, nullable_string_t *, int *);

/*
 * Other functions.
 */

/* Return a pseudo-random number. */
unsigned int randomNumber(unsigned int);

/*
 * Debugging functions and macros.
 */

#ifdef ENABLE_DEBUGGING
#if __STDC_VERSION__ < 199901L && !defined(__func__)
#if __GNUC__ >= 2
#define __func__ __FUNCTION__
#else
#define __func__ "<unknown>"
#endif
#endif
#define debug(x,...) debugWriteOutput(__func__, __FILE__, __LINE__, x, __VA_ARGS__)
#else
#define debug(x,...) /*@-noeffect@*/ do { } while (0) /*@+noeffect@*/
#endif

#define debugShowStringValue(description, item) debug("%s: %s = %s", description, #item, NULL == item ? "(null)" : item)
#define debugShowFragmentValue(description, item, length) debug("%s: %s = %.*s", description, #item, (int) length, NULL == item ? "(null)" : item)
#define debugShowIntegerValue(description, item) debug("%s: %s = %d", description, #item, item)
#define debugShowBooleanValue(description, item) debug("%s: %s = %s", description, #item, item ? "true" : "false")

/*
 * Set the debugging destination file, if debugging is enabled.
 */
void debugSetDestination(const char *);

/*
 * Output debugging information, if debugging is enabled.
 */
void debugWriteOutput(const char *, const char *, int, const char *, ...);

#endif				/* _SCW_INTERNAL_H */
