/*
 * Functions handling the state structure.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/*
 * Allocate a new, empty, state structure.
 */
/*@null@*/ struct scwState *newState(void)
{
	struct scwState *state;

	state = malloc(sizeof(*state));
	if (NULL == state) {
		perror(PACKAGE_NAME);
		return NULL;
	}

	memset(state, 0, sizeof(*state));

	state->continueWithoutCheckLock = true;
	state->continueWithoutItemLock = true;
	state->continueWithoutMetrics = true;
	state->continueWithoutOutputFile = true;

	return state;
}


/*
 * Free any allocated memory in the state, and clear it completely.
 */
void clearState( /*@null@ */ struct scwState *state)
{
	if (NULL == state)
		return;

	/*@-keeptrans@ */
	if (NULL != state->strings) {
		size_t stringIndex;

		/* Free all allocated strings. */
		for (stringIndex = 0; stringIndex < state->countStrings; stringIndex++) {
			/*@-unqualifiedtrans@ */
			free(state->strings[stringIndex]);
			state->strings[stringIndex] = NULL;
			/*@+unqualifiedtrans@ */
		}

		/* Free the array of allocated strings itself. */
		free(state->strings);
		state->strings = NULL;
		state->stringsArraySize = 0;
		state->countStrings = 0;
	}

	if (NULL != state->arguments) {
		/* Free the array of command line arguments. */
		free(state->arguments);
		state->arguments = NULL;
		state->argumentCount = 0;
	}
	/*@+keeptrans@ */

	memset(state, 0, sizeof(*state));
}


/*
 * Set the global defaults that should be in place before any configuration
 * files or items are loaded.
 */
void setGlobalDefaults(struct scwState *state)
{
	/* Use the preprocessor to reduce repetitive commands. */
#define defaultGlobalSetting(x) (void) parseSetting(x, strlen(x), &(state->globalSettings), SCW_SOURCE_GLOBAL_CONFIG, NULL, 0)	/* flawfinder: ignore */
	/*
	 * flawfinder warns about using strlen() with possibly unterminated
	 * strings, but these are all static strings which are definitely
	 * null-terminated, so we can turn off the warning.
	 */
	defaultGlobalSetting("ItemsDir = " SYSCONFDIR "/" PACKAGE_NAME "/items/{USER}");
	defaultGlobalSetting("MetricsDir = " LOCALSTATEDIR "/spool/" PACKAGE_NAME "/{USER}/{ITEM}");
	defaultGlobalSetting("CheckLockFile = " LOCALSTATEDIR "/spool/" PACKAGE_NAME "/{USER}/.lock");
	defaultGlobalSetting("Sendmail = /usr/sbin/sendmail -FCronDaemon -i -odi -oem -oi -t");
	defaultGlobalSetting("TransmitForm = curl -s -S -m \"$SCW_TIMEOUT\" --data-binary \"@$SCW_FILE\" \"$SCW_URL\"");
	defaultGlobalSetting
	    ("TransmitJSON = curl -s -S -m \"$SCW_TIMEOUT\" --data-binary \"@$SCW_FILE\" -H 'Content-Type: application/json' \"$SCW_URL\"");
	defaultGlobalSetting("UserConfigFile = " SYSCONFDIR "/" PACKAGE_NAME "/settings/{USER}.cf");
	defaultGlobalSetting("ItemListFile = " LOCALSTATEDIR "/spool/" PACKAGE_NAME "/items.json");
	/*
	 * NB crontabs are in /etc/cron.d regardless of SYSCONFDIR, since
	 * cron is a separate package.
	 */
	defaultGlobalSetting("CrontabFile = /etc/cron.d/" PACKAGE_NAME);
	defaultGlobalSetting("UpdateLockFile = " LOCALSTATEDIR "/spool/" PACKAGE_NAME "/.update-lock");
	defaultGlobalSetting("SilentConcurrency = yes");
	defaultGlobalSetting("IgnoreOverrun = no");
	defaultGlobalSetting("SilentDependency = no");
	defaultGlobalSetting("SilentConflict = no");
	defaultGlobalSetting("StatusMode = fd");
	defaultGlobalSetting("TimestampUTC = no");
	defaultGlobalSetting("HTTPInterval = 2");
	defaultGlobalSetting("HTTPTimeout = 20");
	defaultGlobalSetting("SharedSecret = ");
	defaultGlobalSetting("EmailMaxBodySize = unlimited");
	defaultGlobalSetting("EmailBodyText = The output is attached.");
	defaultGlobalSetting("EmailAttachmentName = output.txt");
	defaultGlobalSetting("EmailSender = \"(Cron Daemon)\" <{USER}>");
	defaultGlobalSetting("EmailSubject = Cron <{USER}@{HOSTNAME}> {COMMAND}");
	defaultGlobalSetting("ReceiverStrategy = auto");
}


/*
 * If no output map has been defined, define one.  This should be called
 * after all configuration files, item settings, and command-line arguments
 * have been processed.
 */
void setDefaultOutputMap(struct scwState *state)
{
	if (state->globalSettings.countOutputMaps > 0)
		return;
	if (state->userSettings.countOutputMaps > 0)
		return;
	if (state->itemSettings.countOutputMaps > 0)
		return;
	if (state->commandLineSettings.countOutputMaps > 0)
		return;
	defaultGlobalSetting("OutputMap = OES stamped " LOCALSTATEDIR "/log/" PACKAGE_NAME "/{USER}/{ITEM}.log");
}
