/*
 * Functions for the "status" action.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/file.h>
#include <errno.h>


/*
 * Populate the given status structure with details taken from the given
 * metrics directory, returning false on error.
 *
 * The CheckLockFile lock must have been acquired prior to calling this
 * function unless "unlocked" is true.
 *
 * If "unlocked" is true, then isRunning is left false, as the lock is
 * required to be able to safely check whether a run is in progress.
 */
bool loadStatusFromMetrics(struct scwSettingValue *itemMetricsDir, struct scwItemStatus *itemStatus, bool unlocked)
{
	struct stat statBuf;
	int valueFromFile;
	FILE *statusStream;

	memset(itemStatus, 0, sizeof(*itemStatus));

	if (NULL == itemMetricsDir->expandedValue)
		return false;

#define readMetricsTimestamp(string, var) memset(&statBuf, 0, sizeof(statBuf)); \
	if (0 == fileStatAt(itemMetricsDir->expandedValue, itemMetricsDir->expandedLength, STATIC_STRING(string), &statBuf)) { \
		itemStatus->var = (time_t) (statBuf.st_mtime); \
	} else { \
		itemStatus->var = (time_t) 0; \
	}

	/* Read the timestamps. */

	readMetricsTimestamp("disabled", whenDisabled);
	readMetricsTimestamp("prerequisites-met", whenPrerequisitesLastMet);
	readMetricsTimestamp("started", lastStarted);
	readMetricsTimestamp("ended", lastEnded);
	readMetricsTimestamp("succeeded", lastSucceeded);
	readMetricsTimestamp("failed", firstFailed);
	readMetricsTimestamp("overran", firstOverran);
	readMetricsTimestamp("last-status", lastStatusWritten);
	readMetricsTimestamp("delay", whenDelayChosen);

#define readIntegerFromFile(string, var) valueFromFile = fileReadIntegerFromFileAt(itemMetricsDir->expandedValue, itemMetricsDir->expandedLength, STATIC_STRING(string)); \
	if (valueFromFile > 0) { \
		itemStatus->var = (unsigned int) valueFromFile; \
	} else { \
		itemStatus->var = 0; \
	}

	/* Read the integer values. */

	readIntegerFromFile("success-interval", successInterval);
	readIntegerFromFile("delay", currentDelay);
	readIntegerFromFile("run-time", lastRunTime);

	valueFromFile =
	    fileReadIntegerFromFileAt(itemMetricsDir->expandedValue, itemMetricsDir->expandedLength,
				      STATIC_STRING("pid"));
	if (valueFromFile > 0) {
		itemStatus->pid = (pid_t) valueFromFile;
	} else {
		itemStatus->pid = 0;
	}

	/* Read the last status message. */
	statusStream =
	    fileOpenStreamForReadAt(itemMetricsDir->expandedValue, itemMetricsDir->expandedLength,
				    STATIC_STRING("last-status"));
	if (NULL != statusStream) {
		char *newline;

		memset(itemStatus->lastStatus, 0, sizeof(itemStatus->lastStatus));
		if (NULL == fgets(itemStatus->lastStatus, (int) (sizeof(itemStatus->lastStatus)), statusStream))
			itemStatus->lastStatus[0] = '\0';

		newline = strchr(itemStatus->lastStatus, '\n');
		if (NULL != newline)
			newline[0] = '\0';

		(void) fclose(statusStream);
	}

	if (itemStatus->whenDisabled != 0)
		itemStatus->isDisabled = true;

	if (itemStatus->whenPrerequisitesLastMet != 0)
		itemStatus->hasPrerequisitesMet = true;

	if (itemStatus->lastStarted == 0)
		itemStatus->hasNeverRun = true;

	if (!unlocked)
		itemStatus->isRunning = itemIsRunning(itemMetricsDir);

	if (itemStatus->firstOverran != 0)
		itemStatus->isOverrunning = true;

	if (itemStatus->hasPrerequisitesMet && (itemStatus->lastSucceeded != 0) && (itemStatus->successInterval > 0)) {
		double timeSinceLastSucceeded;

		timeSinceLastSucceeded = difftime(time(NULL), itemStatus->lastSucceeded);
		if (timeSinceLastSucceeded > 0.0
		    && (((unsigned int) timeSinceLastSucceeded) > itemStatus->successInterval)) {
			itemStatus->tooLongSinceLastSuccess = true;
		}
	}

	if (itemStatus->firstFailed != 0)
		itemStatus->lastRunFailed = true;

	if (itemStatus->currentDelay != 0)
		itemStatus->willRunAfterDelay = true;

	return true;
}


/*
 * Show the current status of the selected item, returning a sum of
 * SCW_EXIT_* values:
 *
 *   SCW_EXIT_STATUS_NONEXISTENT if the item has no settings/script file
 *   SCW_EXIT_STATUS_DISABLED if the item is disabled
 *   SCW_EXIT_STATUS_RUNNING if the item is currently running
 *
 * It may also return any other exit status if there is an error.
 */
int showItemStatus(struct scwState *state)
{
	struct scwSettings combinedSettings;
	int retcode, checkLockDescriptor;
	struct scwItemStatus itemStatus;
	const char *metricsDir;
	size_t metricsDirLength;

	memset(&combinedSettings, 0, sizeof(combinedSettings));
	retcode = combineSettings(state, &combinedSettings);
	if (retcode != 0)
		return retcode;

	if (NULL == state->item)
		return SCW_EXIT_BAD_ARGS;

	/*@-mustfreefresh@ */
	/* splint note: gettext triggers a warning we can't resolve. */

	printf("# %s: %.*s\n", _("Item"), (int) (state->itemLength), state->item);
	printf("\n");

	expandAllRawValues(state, &combinedSettings);
	outputAllSettings(state, &combinedSettings);

	printf("\n");

	/* Ensure there is a CheckLockFile. */

	if (NULL == combinedSettings.checkLockFile.expandedValue) {
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, "no CheckLockFile defined");
		return SCW_EXIT_ERROR;
	}

	metricsDir = combinedSettings.metricsDir.expandedValue;
	metricsDirLength = combinedSettings.metricsDir.expandedLength;

	/* Ensure that there is a metrics directory. */

	if (NULL == metricsDir) {
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, "no MetricsDir defined");
		return SCW_EXIT_ERROR;
	}

	metricsDir = combinedSettings.metricsDir.expandedValue;
	metricsDirLength = combinedSettings.metricsDir.expandedLength;

	retcode = directoryParentCreate(metricsDir, metricsDirLength);
	if (retcode != 0)
		return retcode;
	retcode = directoryCreate(metricsDir, metricsDirLength);
	if (retcode != 0)
		return retcode;

	/* Acquire the CheckLock. */

	checkLockDescriptor = fileOpenForAppend(combinedSettings.checkLockFile.expandedValue);
	if (checkLockDescriptor < 0)
		return SCW_EXIT_ERROR;
	if (0 != flock(checkLockDescriptor, LOCK_EX)) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, combinedSettings.checkLockFile.expandedValue,
			strerror(errno));
		(void) close(checkLockDescriptor);
		return SCW_EXIT_ERROR;
	}

	/* Read the metrics. */

	memset(&itemStatus, 0, sizeof(itemStatus));
	if (!loadStatusFromMetrics(&(combinedSettings.metricsDir), &itemStatus, false)) {
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, "failed to read item metrics");
		(void) flock(checkLockDescriptor, LOCK_UN);
		(void) close(checkLockDescriptor);
		return SCW_EXIT_ERROR;
	}

	/* Release the CheckLock. */

	(void) flock(checkLockDescriptor, LOCK_UN);
	(void) close(checkLockDescriptor);

	/* Display the information. */

	printf("# %s %s\n", _("Last successful completion time:"), timeAndDateString(itemStatus.lastSucceeded));
	printf("# %s %s\n", _("Last start time:"), timeAndDateString(itemStatus.lastStarted));
	printf("# %s %s\n", _("Last end time:"), timeAndDateString(itemStatus.lastEnded));

	if (itemStatus.lastRunTime > 0) {
		printf("# %s %s\n", _("Time elapsed during last run:"),
		       timePeriodString((int) (itemStatus.lastRunTime)));
	} else {
		printf("# %s %s\n", _("Time elapsed during last run:"), _("(no run completed yet)"));
	}

	if (itemStatus.lastRunFailed) {
		printf("# ** %s\n", _("The most recent run failed."));
		printf("# ** %s %s\n", _("Time of first failure:"), timeAndDateString(itemStatus.firstFailed));
	}

	if (itemStatus.isOverrunning) {
		printf("# ** %s\n", _("The most recent run took too long, preventing the next scheduled run."));
		printf("# ** %s %s\n", _("Time of first collision with subsequent scheduled run:"),
		       timeAndDateString(itemStatus.firstOverran));
	}

	if (itemStatus.isDisabled) {
		printf("# !! %s\n", _("This item is currently disabled."));
		printf("# !! %s %s\n", _("Disabled since:"), timeAndDateString(itemStatus.whenDisabled));
		retcode += SCW_EXIT_STATUS_DISABLED;
	} else {
		printf("# %s\n", _("This item is currently enabled."));
	}
	if (itemStatus.isRunning) {
		printf("# %s\n", _("This item is currently running."));
		if (itemStatus.pid > 0) {
			printf("# %s %u\n", _("Running process ID:"), (unsigned int) (itemStatus.pid));
		} else {
			printf("# %s\n", _("Running process ID is unknown."));
		}
		retcode += SCW_EXIT_STATUS_RUNNING;
	} else {
		printf("# %s\n", _("This item is not currently running."));
	}

	if (itemStatus.willRunAfterDelay) {
		printf("# %s: %u %s\n", _("This item will start after a randomly chosen delay"),
		       itemStatus.currentDelay, _("sec"));
	}

	if ('\0' != itemStatus.lastStatus[0])
		printf("# %s %s\n", _("Last status update:"), itemStatus.lastStatus);

	if (0 != itemStatus.lastStatusWritten)
		printf("# %s %s\n", _("Time of last status update:"), timeAndDateString(itemStatus.lastStatusWritten));

	if (!state->itemHasSettingsFile) {
		printf("# ** %s\n", _("This item has neither a settings file nor a script file in ItemsDir."));
		retcode += SCW_EXIT_STATUS_NONEXISTENT;
	}

	/*@+mustfreefresh@ */

	return retcode;
}
