/*
 * Functions for the "update" action.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <errno.h>


/*
 * Update the crontab and the item list file.
 */
int updateGlobalFiles(struct scwState *state)
{
	struct scwSettings combinedSettings;
	int lockDescriptor;
	size_t crontabTempPathLength;
	char *crontabTempPath = NULL;
	int crontabTempDescriptor = -1;
	FILE *crontabTempStream = NULL;
	size_t itemListTempPathLength;
	char *itemListTempPath = NULL;
	int itemListTempDescriptor = -1;
	FILE *itemListTempStream = NULL;
	bool systemCrontab;
	char **usernameArray = NULL;
	size_t usernameCount = 0;
	size_t usernameIndex;
	bool firstItem = true;
	char jsonBuffer[SCW_MAX_JSONSTRING];	/* flawfinder: ignore */

	/* flawfinder - jsonBuffer is zeroed and bounded. */
	memset(jsonBuffer, 0, sizeof(jsonBuffer));

	/* Combine the global and command-line settings. */
	memset(&combinedSettings, 0, sizeof(combinedSettings));
	if (0 != combineSettings(state, &combinedSettings))
		return SCW_EXIT_ERROR;

	/* Expand ItemListFile, CrontabFile, UpdateLockFile. */
	/*@-mustfreefresh@ *//* gettext causes splint warnings. */
	expandRawValue(state, &(combinedSettings.itemListFile));
	if (NULL == combinedSettings.itemListFile.expandedValue) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, "ItemListFile", _("no value set"));
		return SCW_EXIT_BAD_CONFIG;
	}
	expandRawValue(state, &(combinedSettings.crontabFile));
	if (NULL == combinedSettings.crontabFile.expandedValue) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, "CrontabFile", _("no value set"));
		return SCW_EXIT_BAD_CONFIG;
	}
	expandRawValue(state, &(combinedSettings.checkLockFile));
	if (NULL == combinedSettings.checkLockFile.expandedValue) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, "UpdateLockFile", _("no value set"));
		return SCW_EXIT_BAD_CONFIG;
	}
	/*@+mustfreefresh@ */

	/* Lock the UpdateLockFile. */
	lockDescriptor = fileOpenForAppend(combinedSettings.checkLockFile.expandedValue);
	if (lockDescriptor < 0)
		return SCW_EXIT_ERROR;
	if (0 != flock(lockDescriptor, LOCK_EX)) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, combinedSettings.checkLockFile.expandedValue,
			strerror(errno));
		(void) close(lockDescriptor);
		return SCW_EXIT_ERROR;
	}

	/* Set the umask to restrict mkstemp() file permissions. */
	(void) umask(0077);		    /* flawfinder: ignore */
	/* flawfinder says to ensure this is restrictive - it is. */

	/* Open temporary files for the new crontab and item list file contents. */

	/* 7 = strlen(".XXXXXX"), the template suffix. */
	crontabTempPathLength = combinedSettings.crontabFile.expandedLength + 7;
	crontabTempPath = stringCopy(state, NULL, crontabTempPathLength);
	if (NULL == crontabTempPath) {
		(void) flock(lockDescriptor, LOCK_UN);
		(void) close(lockDescriptor);
		return SCW_EXIT_ERROR;
	}
	memcpy(crontabTempPath, combinedSettings.crontabFile.expandedValue,	/* flawfinder: ignore */
	       combinedSettings.crontabFile.expandedLength);
	memcpy(crontabTempPath + combinedSettings.crontabFile.expandedLength,	/* flawfinder: ignore */
	       ".XXXXXX", 7);
	/* flawfinder - the buffer was allocated to be the right size. */

	/*@-unrecog@ *//* splint doesn't know about mkstemp(). */
	crontabTempDescriptor = mkstemp(crontabTempPath);	/* flawfinder: ignore */
	/*@+unrecog@ */
	if (crontabTempDescriptor < 0) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, crontabTempPath, strerror(errno));
		(void) flock(lockDescriptor, LOCK_UN);
		(void) close(lockDescriptor);
		return SCW_EXIT_ERROR;
	}
	crontabTempStream = fdopen(crontabTempDescriptor, "w");
	if (NULL == crontabTempStream) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, crontabTempPath, strerror(errno));
		(void) close(crontabTempDescriptor);
		(void) remove(crontabTempPath);
		(void) flock(lockDescriptor, LOCK_UN);
		(void) close(lockDescriptor);
		return SCW_EXIT_ERROR;
	}

	/* As above, 7 = strlen(".XXXXXX"), the template suffix. */
	itemListTempPathLength = combinedSettings.itemListFile.expandedLength + 7;
	itemListTempPath = stringCopy(state, NULL, itemListTempPathLength);
	if (NULL == itemListTempPath) {
		(void) fclose(crontabTempStream);
		(void) remove(crontabTempPath);
		(void) flock(lockDescriptor, LOCK_UN);
		(void) close(lockDescriptor);
		return SCW_EXIT_ERROR;
	}
	memcpy(itemListTempPath, combinedSettings.itemListFile.expandedValue,	/* flawfinder: ignore */
	       combinedSettings.itemListFile.expandedLength);
	memcpy(itemListTempPath + combinedSettings.itemListFile.expandedLength,	/* flawfinder: ignore */
	       ".XXXXXX", 7);
	/* flawfinder - the buffer was sized to fit these strings. */

	/*@-unrecog@ *//* splint doesn't know about mkstemp(). */
	itemListTempDescriptor = mkstemp(itemListTempPath);	/* flawfinder: ignore */
	/*@+unrecog@ */
	if (itemListTempDescriptor < 0) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, itemListTempPath, strerror(errno));
		(void) fclose(crontabTempStream);
		(void) remove(crontabTempPath);
		(void) flock(lockDescriptor, LOCK_UN);
		(void) close(lockDescriptor);
		return SCW_EXIT_ERROR;
	}
	itemListTempStream = fdopen(itemListTempDescriptor, "w");
	if (NULL == itemListTempStream) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, itemListTempPath, strerror(errno));
		(void) fclose(crontabTempStream);
		(void) remove(crontabTempPath);
		(void) close(itemListTempDescriptor);
		(void) remove(itemListTempPath);
		(void) flock(lockDescriptor, LOCK_UN);
		(void) close(lockDescriptor);
		return SCW_EXIT_ERROR;
	}

	/* Determine whether to use system crontab format. */
	systemCrontab = false;
	if (state->allUsers
	    || (combinedSettings.crontabFile.expandedLength > 12
		&& stringStartsWith(combinedSettings.crontabFile.expandedValue, "/etc/cron.d/")))
		systemCrontab = true;

	/* Open the item list array. */
	fprintf(itemListTempStream, "[");

	/* Enumerate the users to find items for. */

	if (state->allUsers) {
		enumerateAllUsers(state, &usernameArray, &usernameCount);
	} else {
		usernameArray = (char **) stringCopy(state, NULL, sizeof(char *) - 1);
		if (NULL == usernameArray) {
			(void) fclose(crontabTempStream);
			(void) fclose(itemListTempStream);
			(void) remove(crontabTempPath);
			(void) remove(itemListTempPath);
			(void) flock(lockDescriptor, LOCK_UN);
			(void) close(lockDescriptor);
			return SCW_EXIT_ERROR;
		}
		/*@-dependenttrans@ */
		usernameArray[0] = state->username;
		/*@+dependenttrans@ */
		usernameCount = 1;
	}

	for (usernameIndex = 0; NULL != usernameArray && usernameIndex < usernameCount; usernameIndex++) {
		char *originalUsername;
		size_t originalUsernameLength;
		char *username;
		size_t usernameLength;
		char **itemArray = NULL;
		size_t itemCount = 0;
		size_t itemIndex;

		username = usernameArray[usernameIndex];
		if (NULL == username)
			continue;
		usernameLength = strlen(username);	/* flawfinder: ignore */
		/*
		 * flawfinder warns of strlen() with non-null-terminated
		 * strings; the array contains strings we have explicitly
		 * null terminated.
		 */

		originalUsername = state->username;
		originalUsernameLength = state->usernameLength;
		state->username = username;
		state->usernameLength = usernameLength;

		/* Load the per-user configuration for this user. */
		memset(&(state->userSettings), 0, sizeof(state->userSettings));
		if (0 != loadUserConfig(state)) {
			state->username = originalUsername;
			state->usernameLength = originalUsernameLength;
			continue;
		}

		/* Find the items for this user. */
		enumerateUserItems(state, &itemArray, &itemCount);

		/* Load each item and display its information. */
		for (itemIndex = 0; NULL != itemArray && itemIndex < itemCount; itemIndex++) {
			struct scwSettings combinedItemSettings;

			state->item = itemArray[itemIndex];
			if (NULL == state->item)
				continue;
			state->itemLength = strlen(state->item);	/* flawfinder: ignore */
			/*
			 * flawfinder warns of strlen() with
			 * non-null-terminated strings; the array contains
			 * strings we have explicitly null terminated.
			 */

			debug("%s: %.*s / %.*s", "considering item", state->usernameLength, state->username,
			      state->itemLength, state->item);

			/* Load the item's settings. */
			memset(&(state->itemSettings), 0, sizeof(state->itemSettings));
			if (0 != loadCurrentItemSettings(state))
				continue;

			/* Combine and expand the settings for this item. */
			memset(&combinedItemSettings, 0, sizeof(combinedItemSettings));
			if (0 != combineSettings(state, &combinedItemSettings))
				continue;
			expandAllRawValues(state, &combinedItemSettings);

			debug("%s=%d", "item schedule count", state->itemSettings.countSchedules);
			debug("%s=%d", "combined schedule count", combinedItemSettings.countSchedules);
#ifdef ENABLE_DEBUGGING
			debugOutputAllSettings(state->item, &combinedSettings);
#endif

			/* Write the crontab information. */
			if (combinedItemSettings.countSchedules > 0
			    && NULL != combinedItemSettings.command.expandedValue) {
				size_t scheduleIndex;

				fprintf(crontabTempStream, "# (%.*s:%.*s)", (int) (state->usernameLength),
					state->username, (int) (state->itemLength), state->item);
				if (NULL != combinedItemSettings.description.expandedValue
				    && combinedItemSettings.description.expandedLength < 1) {
					fprintf(crontabTempStream, "%.*s",
						(int) (combinedItemSettings.description.expandedLength),
						combinedItemSettings.description.expandedValue);
				}
				fprintf(crontabTempStream, "\n");

				for (scheduleIndex = 0; scheduleIndex < combinedItemSettings.countSchedules;
				     scheduleIndex++) {
					if (NULL == combinedItemSettings.schedule[scheduleIndex].expandedValue)
						continue;
					fprintf(crontabTempStream, "%s ",
						combinedItemSettings.schedule[scheduleIndex].expandedValue);
					if (systemCrontab)
						fprintf(crontabTempStream, "%.*s ", (int) (state->usernameLength),
							state->username);
					fprintf(crontabTempStream, "%s run %.*s\n", PACKAGE_NAME,
						(int) (state->itemLength), state->item);
				}
			}

			/* Write the item list information. */
			if (firstItem) {
				fprintf(itemListTempStream, "\n");
				firstItem = false;
			} else {
				fprintf(itemListTempStream, ",\n");
			}
			fprintf(itemListTempStream, "{");
			fprintf(itemListTempStream, "\"%s\":\"%s\"", "username",
				jsonEscapeString(jsonBuffer, sizeof(jsonBuffer), state->username,
						 state->usernameLength, NULL));
			fprintf(itemListTempStream, ",\"%s\":\"%s\"", "item",
				jsonEscapeString(jsonBuffer, sizeof(jsonBuffer), state->item, state->itemLength, NULL));
			fprintf(itemListTempStream, ",\"%s\":\"%s\"", "description",
				jsonEscapeString(jsonBuffer, sizeof(jsonBuffer),
						 combinedItemSettings.description.expandedValue,
						 combinedItemSettings.description.expandedLength, NULL));
			fprintf(itemListTempStream, ",\"%s\":\"%s\"", "metricsDir",
				jsonEscapeString(jsonBuffer, sizeof(jsonBuffer),
						 combinedItemSettings.metricsDir.expandedValue,
						 combinedItemSettings.metricsDir.expandedLength, NULL));
			fprintf(itemListTempStream, ",\"%s\":%u", "successInterval",
				combinedItemSettings.numSuccessInterval);
			fprintf(itemListTempStream, "}");
		}

		state->item = NULL;
		state->itemLength = 0;

		state->username = originalUsername;
		state->usernameLength = originalUsernameLength;
	}

	/* Close the item list array. */
	fprintf(itemListTempStream, "\n]\n");

	/* Close the output files and rename them to their final destinations. */

	if (0 != fclose(crontabTempStream))
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, crontabTempPath, strerror(errno));
	if (0 != rename(crontabTempPath, combinedSettings.crontabFile.expandedValue)) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, combinedSettings.crontabFile.expandedValue,
			strerror(errno));
		(void) remove(crontabTempPath);
	}

	/* The item list should be world readable. */
	if (0 != fchmod(itemListTempDescriptor, 0644))
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, itemListTempPath, strerror(errno));
	if (0 != fclose(itemListTempStream))
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, itemListTempPath, strerror(errno));
	if (0 != rename(itemListTempPath, combinedSettings.itemListFile.expandedValue)) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, combinedSettings.itemListFile.expandedValue,
			strerror(errno));
		(void) remove(itemListTempPath);
	}

	/* Release the lock. */
	(void) flock(lockDescriptor, LOCK_UN);
	(void) close(lockDescriptor);

	return 0;
}
