/*
 * Output a string, with word wrapping, supporting wide characters if
 * possible.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#if defined(ENABLE_NLS) && defined(HAVE_WCHAR_H)
#include <wchar.h>
#if defined(HAVE_WCTYPE_H)
#include <wctype.h>
#endif
#endif


/*
 * Return the number of display columns needed to show the given string.
 *
 * To do this, we convert it to a wide character string, and use the wide
 * character display width function "wcswidth()" on it.
 *
 * If NLS is disabled, or the string cannot be converted, this is just the
 * same as "strlen()".
 */
size_t calculateDisplayedWidth(const char *string)
{
	size_t width;

#if defined(ENABLE_NLS) && defined(HAVE_WCHAR_H)
	size_t countWideChars;
	size_t bufferSize;
	wchar_t *wideString;

	if (NULL == string)
		return 0;

	/*@-nullpass@ */
	/*
	 * splint note: mbstowcs() manual page on Linux explicitly says it
	 * takes NULL.
	 */
	countWideChars = mbstowcs(NULL, string, 0);
	/*@+nullpass@ */
	if (countWideChars == (size_t) -1) {
		debug("%s: %s: %s", "mbstowcs", string, strerror(errno));
		return strlen(string);	    /* flawfinder: ignore */
		/*
		 * flawfinder rationale: we have already checked for NULL,
		 * and we don't know the size of the originating buffer so
		 * can't use strnlen(); it is up to the caller to provide a
		 * terminated string.
		 */
	}

	bufferSize = sizeof(*wideString) * (1 + countWideChars);
	wideString = malloc(bufferSize);
	if (NULL == wideString) {
		perror("malloc");
		return strlen(string);	    /* flawfinder: ignore */
		/* flawfinder rationale: see above. */
	}
	memset(wideString, 0, bufferSize);

	if (mbstowcs(wideString, string, 1 + countWideChars) == (size_t) -1) {
		debug("%s: %s: %s", "mbstowcs", string, strerror(errno));
		width = strlen(string);	    /* flawfinder: ignore */
		/* flawfinder rationale: see above. */
	} else if (NULL != wideString) {
		/*@-unrecog @ */
		/* splint seems unable to see the prototype. */
		width = wcswidth(wideString, countWideChars);
		/*@+unrecog @ */
	} else {
		width = 0;
	}

	free(wideString);

#else				/* ! defined(ENABLE_NLS) && defined(HAVE_WCHAR_H) */
	if (NULL == string)
		return 0;

	width = strlen(string);		    /* flawfinder: ignore */
	/* flawfinder rationale: see above. */
#endif				/* defined(ENABLE_NLS) && defined(HAVE_WCHAR_H) */

	return width;
}


/*
 * Output a string to "stream", word wrapping to "wrapWidth" characters, and
 * left-padding any new lines after the first one with "leftMargin" spaces.
 *
 * This is a 7-bit ASCII version of outputWordWrap() below - it does not
 * understand multi-byte characters, so it would count those incorrectly.
 */
static void nonWideOutputWordWrap(FILE * stream, const char *string, size_t wrapWidth, size_t leftMargin)
{
	const char *start;
	const char *end;

	if (NULL == string)
		return;

	start = string;

	while (strlen(start) > wrapWidth) { /* flawfinder: ignore */
		/* flawfinder rationale: see above. */

		/*
		 * Find the end of the last word that will fit on this line.
		 */
		end = start + wrapWidth;
		while ((end > start) && (end[0] != ' '))
			end--;
		if (end == start) {
			/*
			 * No appropriate space found, so just display all
			 * the characters up to the wrap width.
			 */
			end = start + wrapWidth;
		} else {
			/*
			 * We've moved backwards to before the space, so
			 * move forwards again into the space.
			 */
			end++;
		}

		/* Output the selected part of the string. */
		fprintf(stream, "%.*s", (int) (end - start), start);

		/* Ensure we never get stuck. */
		if (end == start)
			end++;

		/*
		 * Move the start position forward to where we just
		 * displayed up to.
		 */
		start = end;

		/*
		 * If there's anything left to display, output a new line,
		 * plus the spaces for the left margin.
		 */
		if (start[0] != '\0')
			fprintf(stream, "\n%*s", (int) leftMargin, "");
	}

	/* Output whatever remains. */
	fprintf(stream, "%s", start);
}


/*
 * Output a string to "stream", word wrapping to "wrapWidth" display
 * character positions, and left-padding any new lines after the first one
 * with "leftMargin" spaces.
 *
 * Wide characters are handled if NLS is enabled, but if they can't be, this
 * falls back to the version above, which just counts bytes as characters.
 */
void outputWordWrap(FILE * stream, const char *string, size_t wrapWidth, size_t leftMargin)
{
#if defined(ENABLE_NLS) && defined(HAVE_WCHAR_H)
	size_t stringWidth;
	size_t bufferSize;
	wchar_t *wideString;
	size_t startPosition, endPosition, charsRemaining;

	if (NULL == string)
		return;

	/*@-nullpass@ */
	/* splint note: see earlier mbstowcs() call. */
	stringWidth = mbstowcs(NULL, string, 0);
	/*@+nullpass@ */
	if (stringWidth == (size_t) -1) {
		/*
		 * Unable to determine how many wide characters the
		 * multibyte string will conver to - fall back to the non
		 * multibyte version of this function.
		 */
		debug("%s: %s: %s", "mbstowcs", string, strerror(errno));
		nonWideOutputWordWrap(stream, string, wrapWidth, leftMargin);
		return;
	}

	bufferSize = sizeof(*wideString) * (1 + stringWidth);
	wideString = malloc(bufferSize);
	if (NULL == wideString) {
		/*
		 * Unable to allocate a buffer to hold the wide-character
		 * version of the supplied multibyte string - fall back to
		 * the non multibyte version of this function.
		 */
		perror("malloc");
		nonWideOutputWordWrap(stream, string, wrapWidth, leftMargin);
		return;
	}
	memset(wideString, 0, bufferSize);

	if (mbstowcs(wideString, string, 1 + stringWidth) == (size_t) -1) {
		/*
		 * Failed to perform the conversion - fall back, as above.
		 */
		debug("%s: %s: %s", "mbstowcs", string, strerror(errno));
		free(wideString);
		nonWideOutputWordWrap(stream, string, wrapWidth, leftMargin);
		return;
	}

	startPosition = 0;
	charsRemaining = stringWidth;

	while (charsRemaining > 0 && wcswidth(&(wideString[startPosition]), charsRemaining) > wrapWidth) {
		size_t nextLineStartPosition;

		/*
		 * Find the end of the last word that will fit on this line.
		 */
		endPosition = startPosition + wrapWidth;
		while ((endPosition > startPosition) && (!iswspace(wideString[endPosition])))
			endPosition--;
		if (endPosition == startPosition) {
			/*
			 * No appropriate space found, so just display all
			 * the characters up to the wrap width.
			 */
			endPosition = startPosition + wrapWidth;
		} else {
			/*
			 * We've moved backwards to before the space, so
			 * move forwards again into the space.
			 */
			endPosition++;
		}

		/*
		 * Record where the next line starts - if it looks like it
		 * would start where we already started this line, move on
		 * one, to avoid getting stuck.
		 */
		nextLineStartPosition = endPosition;
		if (endPosition == startPosition)
			nextLineStartPosition++;

		/*
		 * Convert each wide character to a multibyte string, so it
		 * can be written to the output.
		 */
		while (startPosition < endPosition && startPosition < stringWidth) {
			char multiByteString[MB_CUR_MAX + 1];	/* flawfinder: ignore */
			/*
			 * flawfinder rationale: array is explicitly
			 * cleared, large enough according to the wctomb()
			 * manual, and we explicitly terminate the string.
			 */
			memset(multiByteString, 0, MB_CUR_MAX + 1);
			if (wctomb(multiByteString, wideString[startPosition]) >= 0) {
				multiByteString[MB_CUR_MAX] = '\0';
				fprintf(stream, "%s", multiByteString);
			}
			startPosition++;
		}

		startPosition = nextLineStartPosition;

		/*
		 * If there's anything left to display, output a new line,
		 * plus the spaces for the left margin.
		 */
		if (startPosition < stringWidth)
			fprintf(stream, "\n%*s", (int) leftMargin, "");

		charsRemaining = stringWidth - startPosition;
	}

	/*
	 * Output whatever remains.  As in the loop above, convert each wide
	 * character in turn to a multibyte string to output it.
	 */
	while (startPosition < stringWidth) {
		char multiByteString[MB_CUR_MAX + 1];	/* flawfinder: ignore */
		/* flawfinder rationale as above. */
		memset(multiByteString, 0, MB_CUR_MAX + 1);
		if (wctomb(multiByteString, wideString[startPosition]) >= 0) {
			multiByteString[MB_CUR_MAX] = '\0';
			fprintf(stream, "%s", multiByteString);
		}
		startPosition++;
	}

	free(wideString);

#else				/* ! defined(ENABLE_NLS) && defined(HAVE_WCHAR_H) */
	nonWideOutputWordWrap(stream, string, wrapWidth, leftMargin);
#endif				/* defined(ENABLE_NLS) && defined(HAVE_WCHAR_H) */
}
