/*
 * Header for functions common to all programs.
 *
 * Copyright 2014, 2021, 2023, 2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later <https://gnu.org/licenses/gpl.html>.
 */

#ifndef COMMON_H
#define COMMON_H 1

#ifndef _STDIO_H
#include <stdio.h>
#endif	/* _STDIO_H */

#ifndef _STDINT_H
#include <stdint.h>
#endif	/* _STDINT_H */

#ifndef _STDDEF_H
#include <stddef.h>
#endif	/* _STDDEF_H */

#ifndef _STDBOOL_H
#include <stdbool.h>
#endif	/* _STDBOOL_H */

#define COPYRIGHT_YEAR "2025"
#define COPYRIGHT_HOLDER "Andrew Wood"

#define ENABLE_SETPROCTITLE 1

/*@-exportlocal@*/
extern bool debugging_enabled;		 /* global flag to enable debugging */
extern bool using_syslog;		 /* global flag to report to syslog */
extern /*@observer@*/ char *common_program_name;	 /* set this to program leafname */
extern int error_count;			 /* global error counter from error() */

/* String pointer that is allowed to be NULL. */
typedef /*@null@ */ char * nullable_string;

/* Chain-copy a string with truncation - from string_copying(7). */
/*@dependent@ */ char *stpecpy(char *dst, /*@out@ */ char *end, const char *restrict src);

/* Comparison function which works around inlined strcmp(). */
int compare_with_strcmp(const void *a, const void *b);

#ifdef ENABLE_DEBUGGING
# if __STDC_VERSION__ < 199901L && !defined(__func__)
#  if __GNUC__ >= 2
#   define __func__ __FUNCTION__
#  else
#   define __func__ "<unknown>"
#  endif
# endif
# define debug(x,...) _debug(__func__, __FILE__, __LINE__, x, __VA_ARGS__)
#else
# define debug(x,...) /*@-noeffect@*/ do { } while (0) /*@+noeffect@*/
#endif
void _debug(const char *, const char *, int, const char *, ...);
void error(const char *, ...);
void die(const char *, ...);

size_t ds_leafname_pos(/*@null@*/ char *pathname);
/*@dependent@*/ /*@null@*/ char *ds_leafname(/*@null@*/ char *pathname);
int ds_tmpfile(/*@null@*/ char *pathname, /*@null@*/ nullable_string *tmpnameptr);


#if ENABLE_SETPROCTITLE
void initproctitle(int, char **);
void setproctitle(const char *, ...);
#else				/* ENABLE_SETPROCTITLE */
#define initproctitle(x,y)
#define setproctitle(x,...)
#endif				/* ENABLE_SETPROCTITLE */
/*@null@*/ char *xstrdup(const char *);
void create_parent_dirs(const char *);

/*
 * Functions related to word wrapping and command-line help text.
 */

/* Read the size of the terminal. */
bool readTerminalSize(FILE *, /*@null@*/ size_t *, /*@null@*/ size_t *);

/* Determine how many display columns a string will occupy. */
size_t calculateDisplayedWidth(const char *);

/* Output a string with word wrapping and a left margin. */
void outputWordWrap(FILE *, const char *, size_t, size_t);

struct parameterDefinition {
	/*@null@ */ const char *shortOption;
	/*@null@ */ const char *longOption;
	/*@null@ *//*@observer@ */ const char *parameterArgument;
	/*@null@ *//*@observer@ */ const char *description;
	struct {
		/*
		 * Structure holding the width, in display character
		 * positions, of the individual parts of the description of
		 * each parameter - calculated after initialisation, so that
		 * translation can be performed first.
		 */
		size_t shortOption;
		size_t longOption;
		size_t parameterArgument;
		size_t description;
	} width;
};

/* Output parameter definitions, wrapped to a specific width. */
void showParameterDefinitions(FILE *, size_t, struct parameterDefinition *);

/*@+exportlocal@*/

#endif	/* COMMON_H */
