/*
 * Configuration parser and main loop to manage multiple concurrent
 * synchronisation sets, providing an interface to the continual
 * synchronisation function continual_sync in sync.c.
 *
 * Copyright 2014, 2021, 2023, 2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later <https://gnu.org/licenses/gpl.html>.
 */

#include "config.h"
#include "common.h"
#include "sync.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdint.h>
#include <getopt.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <wordexp.h>
#include <fnmatch.h>
#ifndef SPLINT
/* splint 3.1.2 chokes on syslog.h */
#include <syslog.h>
#endif


/* List of config sections. */
static struct sync_set_s config_sections[MAX_CONFIG_SECTIONS];
static int config_sections_count = 0;

/* List of config sections chosen on the command line. */
static
 /*@null@ */
 /*@only@ */
char **config_sections_selected = NULL;
static int config_sections_selected_count = 0;

/* PID file if in daemon mode */
static
 /*@null@ */
 /*@only@ */
char *pidfile = NULL;
bool sync_exit_now = false;		 /* exit-now flag (on signal) */


/*
 * Return the index of the config section with the given name, or -1 on
 * failure.
 */
static int find_config_section(const char *name)
{
	int idx;

	for (idx = 0; idx < config_sections_count; idx++) {
		if (NULL == config_sections[idx].name)
			continue;
		if (strcmp(config_sections[idx].name, name) == 0)
			return idx;
	}

	return -1;
}


/*
 * Expand %s, %h etc in the string pointed to by strptr, reallocating the
 * string if necessary, returning nonzero on error (and reporting the
 * error).
 */
static int expand_config_sequences(struct sync_set_s *cf, /*@null@ */ nullable_string * strptr, char *parameter)
{
	size_t orig_string_len = 0;
	size_t new_string_len = 0;
	char *origstr;
	char *newstr;
	size_t idx;

	if (NULL == strptr)
		return 0;
	if (NULL == *strptr)
		return 0;

	origstr = *strptr;
	orig_string_len = strlen(origstr);
	newstr = NULL;

	idx = 0;
	while (idx < orig_string_len) {
		size_t section_length;
		/*@dependent@ */ nullable_string to_append = NULL;
		size_t to_append_len = 0;

		section_length = strcspn(&(origstr[idx]), "%");
		if (section_length > 0) {
			char *extendedstr = NULL;
			extendedstr = realloc(newstr, new_string_len + section_length + 1);
			if (NULL == extendedstr) {
				error("%s: %s", "realloc", strerror(errno));
				return 1;
			}
			newstr = extendedstr;
			strncpy(&(newstr[new_string_len]), &(origstr[idx]), section_length);
			new_string_len += section_length;
			newstr[new_string_len] = '\0';
		}
		idx += section_length;
		if (idx >= orig_string_len)
			break;

		idx++;

		switch (origstr[idx]) {
		case '%':
			to_append = "%";
			to_append_len = 1;
			break;
		case 'n':
			to_append = cf->name;
			to_append_len = strlen(cf->name);
			break;
		case 's':
			if (NULL != cf->source) {
				to_append = cf->source;
				to_append_len = strlen(cf->source);
			}
			break;
		case 'd':
			if (NULL != cf->destination) {
				to_append = strrchr(cf->destination, ':');
				to_append_len = 0;
				if (NULL != to_append) {
					to_append++;
				} else {
					to_append = cf->destination;
				}
				if (NULL != to_append) {
					to_append_len = strlen(to_append);
				}
			}
			break;
		case 'h':
			if (NULL != cf->destination) {
				to_append = cf->destination;
				to_append_len = strcspn(to_append, ":");
				if (to_append_len == strlen(to_append)) {
					to_append = "localhost";
					to_append_len = strlen(to_append);
				}
			}
			break;
		default:
			error("%s: %s: %s: %%%c", cf->name, parameter,
			      _("unknown variable substitution character sequence"), origstr[idx]);
			if (NULL != newstr) {
				free(newstr);
				newstr = NULL;
			}
			return 1;
		}

		idx++;

		if (to_append_len > 0 && NULL != to_append) {
			char *extendedstr;
			extendedstr = realloc(newstr, new_string_len + to_append_len + 1);
			if (NULL == extendedstr) {
				error("%s: %s", "realloc", strerror(errno));
				return 1;
			}
			newstr = extendedstr;
			strncpy(&(newstr[new_string_len]), to_append, to_append_len);
			new_string_len += to_append_len;
			newstr[new_string_len] = '\0';
		}
	}

	if (NULL == newstr)
		return 0;

	if (strcmp(newstr, origstr) == 0) {
		free(newstr);
		return 0;
	}

	debug("(cf) %s: %s: [%s] -> [%s]", cf->name, parameter, origstr, newstr);

	free(origstr);
	*strptr = newstr;

	return 0;
}


/*
 * Return nonzero if the configuration section with the given index is not
 * valid, and report the error.
 *
 * As a side effect, fills in the defaults from DEFAULTS_SECTION and expands
 * the parameters, if this isn't the DEFAULTS_SECTION section.
 */
static int validate_config_section(int idx, int defaults_idx)
{
	int rc = 0;

	if (strcmp(config_sections[idx].name, DEFAULTS_SECTION) == 0) {
		if (NULL != config_sections[idx].source) {
			error("%s: %s: %s", config_sections[idx].name, "source",
			      _("this section must not specify a source directory"));
			rc = 1;
		} else if (NULL != config_sections[idx].destination) {
			error("%s: %s: %s", config_sections[idx].name, "destination",
			      _("this section must not specify a destination directory"));
			rc = 1;
		}
		debug("(cf valid) %d %s: %s", idx, config_sections[idx].name, rc == 0 ? "OK" : "FAILED");
		return rc;
	} else {
		if (NULL == config_sections[idx].source) {
			error("%s: %s: %s", config_sections[idx].name, "source",
			      _("this section has not defined a source directory"));
			rc = 1;
		} else if (NULL == config_sections[idx].destination) {
			error("%s: %s: %s", config_sections[idx].name, "destination",
			      _("this section has not defined a destination directory"));
			rc = 1;
		} else if (NULL != config_sections[idx].section_list_file) {
			error("%s: %s: %s", config_sections[idx].name, "section list file",
			      _("only the default section may define the section list file"));
			rc = 1;
		}
	}

	/*
	 * Fill in the defaults from DEFAULTS_SECTION, if there are any.
	 */
	if (defaults_idx >= 0) {
#define dup_default_string(x) if ((NULL == config_sections[idx].x) && (NULL != config_sections[defaults_idx].x)) { \
config_sections[idx].x = xstrdup(config_sections[defaults_idx].x); \
debug("(cf) %s: %s: %s -> %s", config_sections[idx].name, #x, "using default", config_sections[defaults_idx].x); \
}
		dup_default_string(source_validation);
		dup_default_string(destination_validation);
		dup_default_string(full_marker);
		dup_default_string(partial_marker);
		dup_default_string(change_queue);
		dup_default_string(transfer_list);
		dup_default_string(tempdir);
		dup_default_string(sync_lock);
		dup_default_string(full_rsync_opts);
		dup_default_string(partial_rsync_opts);
		dup_default_string(log_file);
		dup_default_string(status_file);
		/* The section list file is only set in the default section. */
		/* dup_default_string(section_list_file); */
#define copy_default_ulong(x) if ((false == config_sections[idx].set.x) && (false != config_sections[defaults_idx].set.x)) { \
config_sections[idx].x = config_sections[defaults_idx].x; \
debug("(cf) %s: %s: %s -> %lu", config_sections[idx].name, #x, "using default", config_sections[defaults_idx].x); \
}
		copy_default_ulong(full_interval);
		copy_default_ulong(full_retry);
		copy_default_ulong(full_timeout);
		copy_default_ulong(partial_interval);
		copy_default_ulong(partial_retry);
		copy_default_ulong(partial_timeout);
		copy_default_ulong(recursion_depth);
#define copy_default_flag(x) if ((false == config_sections[idx].set.x) && (false != config_sections[defaults_idx].set.x)) { \
config_sections[idx].x = config_sections[defaults_idx].x; \
debug("(cf) %s: %s: %s -> %s", config_sections[idx].name, #x, "using default", config_sections[defaults_idx].x ? "yes" : "no"); \
}
		copy_default_flag(ignore_vanished_files);
		copy_default_flag(track_files);

		if ((0 == config_sections[idx].exclude_count)
		    && (0 != config_sections[defaults_idx].exclude_count)) {
			int eidx;
			debug("(cf) %s: %s", config_sections[idx].name, "using excludes from defaults section");
			config_sections[idx].exclude_count = config_sections[defaults_idx].exclude_count;
			for (eidx = 0; eidx < config_sections[defaults_idx].exclude_count; eidx++) {
				config_sections[idx].excludes[eidx] =
				    xstrdup(config_sections[defaults_idx].excludes[eidx]);
			}
		}
	}

	/*
	 * Expand all %s, %h, etc special sequences before continuing.
	 */
#define expand_sequences(x) if (expand_config_sequences(&(config_sections[idx]), &(config_sections[idx].x), #x) != 0) rc=1;
	expand_sequences(source_validation);
	expand_sequences(destination_validation);
	expand_sequences(full_marker);
	expand_sequences(partial_marker);
	expand_sequences(change_queue);
	expand_sequences(transfer_list);
	expand_sequences(tempdir);
	expand_sequences(sync_lock);
	expand_sequences(full_rsync_opts);
	expand_sequences(partial_rsync_opts);
	expand_sequences(log_file);
	expand_sequences(status_file);
	expand_sequences(section_list_file);

	if (NULL != config_sections[idx].change_queue) {
		struct stat sb;

		memset(&sb, 0, sizeof(sb));

		if (lstat(config_sections[idx].change_queue, &sb) != 0) {
			error("%s: %s: %s: %s", config_sections[idx].name, "change queue",
			      config_sections[idx].change_queue, strerror(errno));
			rc = 1;
		} else if (!S_ISDIR((mode_t) (sb.st_mode))) {
			error("%s: %s: %s: %s", config_sections[idx].name, "change queue",
			      config_sections[idx].change_queue, _("the change queue must be a directory"));
			rc = 1;
		}
	}

	if (NULL != config_sections[idx].tempdir) {
		struct stat sb;
		memset(&sb, 0, sizeof(sb));
		if (lstat(config_sections[idx].tempdir, &sb) != 0) {
			error("%s: %s: %s: %s", config_sections[idx].name, "temporary directory",
			      config_sections[idx].tempdir, strerror(errno));
			rc = 1;
		} else if (!S_ISDIR((mode_t) (sb.st_mode))) {
			error("%s: %s: %s: %s", config_sections[idx].name, "temporary directory",
			      config_sections[idx].tempdir, _("the temporary directory must be a directory"));
			rc = 1;
		}
	}

	if ((0 == config_sections[idx].full_interval)
	    && (0 == config_sections[idx].partial_interval)) {
		error("%s: %s", config_sections[idx].name,
		      _("this section would do nothing, as both the full and partial sync intervals are zero"));
		rc = 1;
	}
#define blank_if_none(x) if ((NULL != config_sections[idx].x) && (strcmp(config_sections[idx].x, "none") == 0)) { \
        free(config_sections[idx].x); \
        config_sections[idx].x = NULL; \
}
	blank_if_none(source_validation);
	blank_if_none(destination_validation);
	blank_if_none(full_marker);
	blank_if_none(partial_marker);
	blank_if_none(change_queue);
	blank_if_none(transfer_list);
	blank_if_none(tempdir);
	blank_if_none(sync_lock);
	blank_if_none(log_file);
	blank_if_none(status_file);
	blank_if_none(section_list_file);

	debug("(cf valid) %d %s: %s", idx, config_sections[idx].name, rc == 0 ? "OK" : "FAILED");

	return rc;
}


/*
 * Parse the given configuration file, returning nonzero on error.
 */
static int parse_config(const char *filename, int depth)
{
	char *linebuf_ptr;
	size_t linebuf_size;
	unsigned int line_number;
	FILE *fptr;
	struct sync_set_s *section;

	if (depth > 3) {
		debug("(cf) %s: %s", filename, "max recursion depth reached - ignoring file");
		return 0;
	}

	debug("(cf) %s: %s", filename, "opening file");

	fptr = fopen(filename, "r");
	if (NULL == fptr) {
		error("%s: %s", filename, strerror(errno));
		return 1;
	}

	linebuf_ptr = NULL;
	linebuf_size = 0;
	line_number = 0;
	section = NULL;

	while (0 == feof(fptr)) {
		ssize_t line_length = 0;
		unsigned long param_ulong;
		char *param_str;
		ssize_t idx;

		errno = 0;
		line_length = getline(&linebuf_ptr, &linebuf_size, fptr);
		if ((line_length < 0) || (NULL == linebuf_ptr)) {
			if (0 != errno)
				error("%s:%u: %s", filename, line_number, strerror(errno));
			break;
		}
		line_number++;

		/*
		 * The length includes the delimiter, and so it should
		 * always be > 0.
		 */
		if (line_length < 1)
			continue;

		/*
		 * Replace the delimiter with \0 to terminate the string
		 * there.
		 */
		line_length--;
		linebuf_ptr[line_length] = '\0';

		param_str = NULL;

		/*
		 * Note that the sscanf() "m" modifier to %[], used below,
		 * is from POSIX.1-2008.  It is available from glibc 2.7,
		 * and doesn't work prior to that.  On CentOS 5, for
		 * instance, it would have to be "%a[]" instead of "%m[]".
		 *
		 * TODO: detect when %m[] isn't available, at compile time.
		 * Or don't, since CentOS 5 went EOL in 2014.
		 */

		if ((sscanf(linebuf_ptr, " [%m[0-9A-Za-z_.-]]", &param_str) == 1) && (NULL != param_str)) {

			/* New section */
			debug("(cf) %s:%u: %s: %s", filename, line_number, "section", param_str);

			if (find_config_section(param_str) >= 0) {
				error("%s:%u: %s: %s", filename, line_number, param_str,
				      _("a section with this name was already defined"));
				free(param_str);
				if (NULL != linebuf_ptr)
					free(linebuf_ptr);
				(void) fclose(fptr);
				return 1;
			}

			if (config_sections_count >= (MAX_CONFIG_SECTIONS - 1)) {
				error("%s: %s: %s", filename, param_str, _("maximum number of sections reached"));
				free(param_str);
				if (NULL != linebuf_ptr)
					free(linebuf_ptr);
				(void) fclose(fptr);
				return 1;
			}

			section = &(config_sections[config_sections_count]);
			config_sections_count++;

			memset(section, 0, sizeof(config_sections[0]));
			section->name = param_str;
			section->full_interval = 86400;
			section->full_retry = 3600;
			section->full_timeout = 0;
			section->partial_interval = 30;
			section->partial_retry = 300;
			section->partial_timeout = 0;
			section->recursion_depth = 20;
			section->ignore_vanished_files = false;
			section->track_files = true;

			continue;

		} else if ((sscanf(linebuf_ptr, " include = %m[^\n]", &param_str)
			    == 1) && (NULL != param_str)) {
			wordexp_t p;
			size_t word_idx;
			int cwdfd;
			char *resolved_path;
			char *ptr;

			/* Include another config file */
			debug("(cf) %s:%u: %s: %s", filename, line_number, "include", param_str);

			/*
			 * Temporarily change directory to the location of
			 * the current configuration file, so that include
			 * paths are relative to the current file.
			 */
			cwdfd = open(".", O_RDONLY | O_DIRECTORY);
			/*@-unrecog@ *//* splint doesn't know about realpath() */
			resolved_path = realpath(filename, NULL);
			/*@+unrecog@ */
			if (NULL == resolved_path) {
				debug("(cf) %s: %s: %s", filename, "realpath", strerror(errno));
			} else {
				ptr = strrchr(resolved_path, '/');
				if (NULL != ptr)
					ptr[0] = '\0';
				if (cwdfd >= 0) {
					int rc;
					rc = chdir(resolved_path);
					debug("(cf) chdir: %s = %d", resolved_path, rc);
				}
				free(resolved_path);
			}

			memset(&p, 0, sizeof(p));

			/*@-compdestroy@ */
			/*
			 * splint doesn't realise that wordfree() releases
			 * the storage within the wordexp_t structure "p",
			 * so reports a memory leak by mistake unless
			 * "-compdestroy" is used to inhibit the warning.
			 */
			if (wordexp(param_str, &p, WRDE_NOCMD) != 0) {
				error("%s:%u: %s: %s: %s", filename, line_number, "include", param_str,
				      strerror(errno));
				(void) fclose(fptr);
				if (cwdfd >= 0) {
					if (0 != fchdir(cwdfd)) {
						error("%s:%u: %s: %s: %s", filename, line_number, "include", param_str,
						      strerror(errno));
					}
					(void) close(cwdfd);
				}
				free(param_str);
				if (NULL != linebuf_ptr)
					free(linebuf_ptr);
				return 1;
			}

			if (cwdfd >= 0) {
				if (0 != fchdir(cwdfd)) {
					error("%s:%u: %s: %s: %s", filename, line_number, "include", param_str,
					      strerror(errno));
				}
				(void) close(cwdfd);
			}

			for (word_idx = 0; word_idx < p.we_wordc; word_idx++) {
				if (access(p.we_wordv[word_idx], F_OK) != 0) {
					debug("(cf) %s: %s: %s", p.we_wordv[word_idx], "skipping", strerror(errno));
					continue;
				}
				if ((fnmatch("*~", p.we_wordv[word_idx], FNM_NOESCAPE) == 0)
				    || (fnmatch("*.rpmsave", p.we_wordv[word_idx], FNM_NOESCAPE) == 0)
				    || (fnmatch("*.rpmorig", p.we_wordv[word_idx], FNM_NOESCAPE) == 0)
				    || (fnmatch("*.rpmnew", p.we_wordv[word_idx], FNM_NOESCAPE) == 0)
				    ) {
					debug("(cf) %s: %s: %s", p.we_wordv[word_idx], "skipping", "ignored");
					continue;
				}
				if (parse_config(p.we_wordv[word_idx], depth + 1) != 0) {
					wordfree(&p);
					if (NULL != linebuf_ptr)
						free(linebuf_ptr);
					return 1;
				}
			}

			wordfree(&p);

			/*@+compdestroy@ */

			free(param_str);

			continue;

		} else if (NULL == section) {

			/* Not in a section at the moment */
			idx = 0;
			while ((idx < line_length) && isspace(linebuf_ptr[idx]))
				idx++;
			if (NULL == strchr("#\r\n", linebuf_ptr[idx])) {
				error("%s:%u: %s", filename, line_number,
				      _("a section declaration is required before any parameters"));
				if (NULL != linebuf_ptr)
					free(linebuf_ptr);
				(void) fclose(fptr);
				return 1;
			}
			continue;
		}

		/*
		 * If we get here, then we're in a section declaration.
		 */

		if (NULL == section) {
			/* There should be no way for this to be reachable. */
			if (NULL != linebuf_ptr)
				free(linebuf_ptr);
			(void) fclose(fptr);
			return 1;
		}

		/*
		 * Strip comments - a hash that's either at the start of the
		 * line or preceded by whitespace.
		 */
		for (idx = 0; idx < line_length; idx++) {
			if ('#' != linebuf_ptr[idx])
				continue;
			if ((idx > 0) && (!isspace(linebuf_ptr[idx - 1])))
				continue;
			linebuf_ptr[idx] = '\0';
			break;
		}

		/*
		 * Strip trailing whitespace.
		 */
		if (line_length > 0) {
			for (idx = line_length - 1; idx > 0; idx--) {
				if (!isspace(linebuf_ptr[idx]))
					break;
				linebuf_ptr[idx] = '\0';
				line_length--;
			}
		}

		/*@-nullderef@ */
		/*@-unrecog@ */

#define cf_string(X, Y) if ((sscanf(linebuf_ptr, " " X, &param_str) == 1) && (NULL != param_str)) { \
debug("(cf) %s:%u: %s = [%s]", filename, line_number, #Y, param_str); \
section->Y = param_str; \
continue; \
}
#define cf_ulong(X, Y) if (sscanf(linebuf_ptr, X, &param_ulong) == 1) { \
debug("(cf) %s:%u: %s = [%lu]", filename, line_number, #Y, param_ulong); \
section->Y = param_ulong; \
section->set.Y = true; \
continue; \
}
#define cf_flag(X, Y) if ((sscanf(linebuf_ptr, " " X, &param_str) == 1) && (NULL != param_str)) { \
section->Y = false; \
if (strncasecmp("yes", param_str, 3) == 0) section->Y = true; \
if (strncasecmp("on", param_str, 2) == 0) section->Y = true; \
debug("(cf) %s:%u: %s = [%s]", filename, line_number, #Y, section->Y ? "yes" : "no"); \
section->set.Y = true; \
free(param_str); \
continue; \
}
		cf_string("source = %m[^\n]", source);
		cf_string("destination = %m[^\n]", destination);
		cf_string("source validation command = %m[^\n]", source_validation);
		cf_string("destination validation command = %m[^\n]", destination_validation);
		cf_ulong("full sync interval = %lu", full_interval);
		cf_ulong("full sync retry = %lu", full_retry);
		cf_ulong("full sync timeout = %lu", full_timeout);
		cf_ulong("partial sync interval = %lu", partial_interval);
		cf_ulong("partial sync retry = %lu", partial_retry);
		cf_ulong("partial sync timeout = %lu", partial_timeout);
		cf_ulong("recursion depth = %lu", recursion_depth);
		cf_string("full sync marker file = %m[^\n]", full_marker);
		cf_string("partial sync marker file = %m[^\n]", partial_marker);
		cf_flag("ignore vanished files = %m[^\n]", ignore_vanished_files);
		cf_flag("track files = %m[^\n]", track_files);
		cf_string("change queue = %m[^\n]", change_queue);
		cf_string("transfer list = %m[^\n]", transfer_list);
		cf_string("temporary directory = %m[^\n]", tempdir);
		cf_string("sync lock = %m[^\n]", sync_lock);
		cf_string("full rsync options = %m[^\n]", full_rsync_opts);
		cf_string("partial rsync options = %m[^\n]", partial_rsync_opts);
		cf_string("log file = %m[^\n]", log_file);
		cf_string("status file = %m[^\n]", status_file);
		cf_string("section list file = %m[^\n]", section_list_file);

		/*@+nullderef@ */
		/*@+unrecog@ */

		/*
		 * splint: the possible NULLs that are highlighted above
		 * would be returned by xstrdup() on allocation failure, but
		 * that function calls die() in that case so would never
		 * actually return.
		 *
		 * Unrecognised identifiers are ignored above because splint
		 * does not recognise strncasecmp().
		 */

		if ((sscanf(linebuf_ptr, " exclude = %m[^\n]", &param_str) == 1) && (NULL != param_str)) {
			debug("(cf) %s:%u: %s = [%s]", filename, line_number, "exclude", param_str);
			if (section->exclude_count >= (MAX_EXCLUDES - 1)) {
				error("%s:%u: %s: %s", filename, line_number, "exclude",
				      _("too many exclusions specified"));
				free(param_str);
				if (NULL != linebuf_ptr)
					free(linebuf_ptr);
				(void) fclose(fptr);
				return 1;
			}
			section->excludes[section->exclude_count++] = param_str;
			continue;
		}

		/*
		 * If we get here, it's either a blank line, a comment, or
		 * an invalid directive.
		 */
		idx = 0;
		while ((idx < line_length) && isspace(linebuf_ptr[idx]))
			idx++;
		if (NULL == strchr("#\r\n", linebuf_ptr[idx])) {
			error("%s:%u: %s", filename, line_number, _("invalid configuration directive"));
			if (NULL != linebuf_ptr)
				free(linebuf_ptr);
			(void) fclose(fptr);
			return 1;
		}
	}

	if (NULL != linebuf_ptr)
		free(linebuf_ptr);

	(void) fclose(fptr);
	return 0;
}


/*
 * Free up memory allocated by parse_options and parse_config.
 */
static void free_options(void)
{
	int cf_idx;
	for (cf_idx = 0; cf_idx < config_sections_count; cf_idx++) {
		int excl_idx;
#define free_and_clear(X) if (NULL != config_sections[cf_idx].X) { \
free(config_sections[cf_idx].X); \
config_sections[cf_idx].X = NULL; \
}
		free_and_clear(name);
		free_and_clear(source);
		free_and_clear(destination);
		free_and_clear(source_validation);
		free_and_clear(destination_validation);
		free_and_clear(full_marker);
		free_and_clear(partial_marker);
		free_and_clear(change_queue);
		free_and_clear(transfer_list);
		free_and_clear(tempdir);
		free_and_clear(sync_lock);
		free_and_clear(full_rsync_opts);
		free_and_clear(partial_rsync_opts);
		free_and_clear(log_file);
		free_and_clear(status_file);
		free_and_clear(section_list_file);
		for (excl_idx = 0; excl_idx < config_sections[cf_idx].exclude_count; excl_idx++) {
			if (NULL != config_sections[cf_idx].excludes[excl_idx]) {
				free(config_sections[cf_idx].excludes[excl_idx]);
				config_sections[cf_idx].excludes[excl_idx]
				    = NULL;
			}
		}
	}
	if (NULL != config_sections_selected)
		free(config_sections_selected);
	config_sections_selected = NULL;
	config_sections_selected_count = 0;
	if (NULL != pidfile) {
		free(pidfile);
		pidfile = NULL;
	}
}


/*
 * Write a string to a file stream, escaping it as JSON.
 */
static void write_json_string(FILE * fptr, /*@null@ */ const char *string)
{
	if (NULL == fptr)
		return;
	if (NULL == string) {
		fprintf(fptr, "null");
		return;
	}

	fprintf(fptr, "\"");

	/* TODO: rewrite this to be multibyte aware so it handles UTF-8. */

	for (; string[0] != '\0'; string++) {
		char next_char = string[0];
		switch (next_char) {
		case '"':
		case '\\':
			fprintf(fptr, "\\%c", next_char);
			break;
		case '\n':
			fprintf(fptr, "\\n");
			break;
		case '\t':
			fprintf(fptr, "\\t");
			break;
		case '\b':
			fprintf(fptr, "\\b");
			break;
		case '\f':
			fprintf(fptr, "\\f");
			break;
		case '\r':
			fprintf(fptr, "\\r");
			break;
		default:
			if (next_char >= ' ' && next_char < (char) 127) {
				fprintf(fptr, "%c", next_char);
			} else {
				unsigned int char_val = (unsigned int) ((unsigned char *) (string))[0];
				fprintf(fptr, "\\u%04x", char_val);
			}
			break;
		}
	}

	fprintf(fptr, "\"");
}


/*
 * Write details of the active sections to the section list file.
 */
static void write_section_list_file(char *savefile, int defaults_idx)
{
	char *temp_filename = NULL;
	int tmpfd, cf_idx;
	FILE *fptr;
	bool first_item;

	if (NULL == savefile)
		return;

	tmpfd = ds_tmpfile(savefile, &temp_filename);
	if ((tmpfd < 0) || (NULL == temp_filename)) {
		return;
	}

	fptr = fdopen(tmpfd, "w");
	if (NULL == fptr) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) close(tmpfd);
		(void) remove(temp_filename);
		free(temp_filename);
		return;
	}

	fprintf(fptr, "[");
	first_item = true;
	for (cf_idx = 0; cf_idx < config_sections_count; cf_idx++) {
		if (!config_sections[cf_idx].selected)
			continue;
		if (cf_idx == defaults_idx)
			continue;

		if (first_item) {
			first_item = false;
		} else {
			fprintf(fptr, ",");
		}

		fprintf(fptr, "\n  {\n");

		fprintf(fptr, "    \"%s\": ", "name");
		write_json_string(fptr, config_sections[cf_idx].name);
		fprintf(fptr, ",\n");

		fprintf(fptr, "    \"%s\": %lu,\n", "fullSyncInterval", config_sections[cf_idx].full_interval);
		fprintf(fptr, "    \"%s\": %lu,\n", "fullSyncRetry", config_sections[cf_idx].full_retry);
		fprintf(fptr, "    \"%s\": %lu,\n", "fullSyncTimeout", config_sections[cf_idx].full_timeout);
		fprintf(fptr, "    \"%s\": %lu,\n", "partialSyncInterval", config_sections[cf_idx].partial_interval);
		fprintf(fptr, "    \"%s\": %lu,\n", "partialSyncRetry", config_sections[cf_idx].partial_retry);
		fprintf(fptr, "    \"%s\": %lu,\n", "partialSyncTimeout", config_sections[cf_idx].partial_timeout);

		fprintf(fptr, "    \"%s\": ", "statusFile");
		write_json_string(fptr, config_sections[cf_idx].status_file);
		fprintf(fptr, "\n");

		fprintf(fptr, "  }");
	}
	fprintf(fptr, "\n]\n");

	if (0 != fchmod(tmpfd, 0644)) {
		error("%s: %s: %s", temp_filename, "fchmod()", strerror(errno));
	}

	if (0 != fclose(fptr)) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		return;
	}

	if (rename(temp_filename, savefile) != 0) {
		error("%s: %s", savefile, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		return;
	}

	free(temp_filename);
}


/*
 * Output program usage information.
 */
static void usage(void)
{
	struct parameterDefinition parameterDefinitions[] = {
		{ "-c", "--config", N_("FILE"),
		 N_("read configuration FILE"),
		 { 0, 0, 0, 0} },
		{ "-D", "--daemon", N_("FILE"),
		 N_("run as a daemon, and write its process ID to FILE"),
		 { 0, 0, 0, 0} },
		{ "", NULL, NULL, NULL, { 0, 0, 0, 0} },
		{ "-h", "--help", NULL,
		 N_("show this help and exit"),
		 { 0, 0, 0, 0} },
		{ "-V", "--version", NULL,
		 N_("show version information and exit"),
		 { 0, 0, 0, 0} },
#ifdef ENABLE_DEBUGGING
		{ "-d", "--debug", NULL,
		 N_("enable debugging"),
		 { 0, 0, 0, 0} },
#endif
		{ NULL, NULL, NULL, NULL, { 0, 0, 0, 0} }
	};
	size_t terminalColumns = 77;
	const char *programDescription;
	const char *bugReportNote;

	(void) readTerminalSize(stdout, &terminalColumns, NULL);
	if (terminalColumns < 6)
		terminalColumns = 6;

	/*@-mustfreefresh@ */
	/*
	 * splint note: the gettext calls made by _() cause memory leak
	 * warnings, but in this case it's unavoidable, and mitigated by the
	 * fact we only translate each string once.
	 */

	printf("%s: %s %s\n", _("Usage"), common_program_name, _("[OPTION...] [SECTION...]"));

	programDescription =
	    _
	    ("Synchronise the directories specified in the given SECTIONS of the configuration file(s), or all sections if nothing is specified.");
	if (NULL != programDescription) {
		outputWordWrap(stdout, programDescription, terminalColumns - 1, 0);
		printf("\n");
	}

	printf("\n");
	showParameterDefinitions(stdout, terminalColumns, parameterDefinitions);
	printf("\n");

	bugReportNote = _("Please report any bugs to:");
	if (NULL != bugReportNote) {
		outputWordWrap(stdout, bugReportNote, terminalColumns - 1, 0);
		printf(" %s\n", PACKAGE_BUGREPORT);
	}
}

/*@+mustfreefresh@ */


/*
 * Parse the command line arguments, and read the configuration files. 
 * Returns 0 on success, -1 if the program should exit immediately without
 * an error, or 1 if the program should exit with an error.
 */
static int parse_options(int argc, char **argv)
{
	/*@-nullassign@ *//* long options can have NULLs */
	struct option long_options[] = {
		{ "help", 0, 0, (int) 'h' },
		{ "version", 0, 0, (int) 'V' },
		{ "config", 1, 0, (int) 'c' },
		{ "daemon", 1, 0, (int) 'D' },
		{ "debug", 0, 0, (int) 'd' },
		{ 0, 0, 0, 0 }
	};
	/*@+nullassign@ */
	int option_index = 0;
	char *short_options = "hVc:D:d";
	int c;
	bool config_specified = false;

	config_sections_selected_count = 0;
	config_sections_selected = calloc((size_t) (argc + 1), sizeof(char *));
	if (NULL == config_sections_selected) {
		die("%s", strerror(errno));
		return 1;
	}

	do {
		c = getopt_long(argc, argv, short_options, long_options, &option_index);
		if (c < 0)
			continue;

		/*
		 * Parse each command line option.
		 */
		switch (c) {
		case 'h':
			usage();
			free_options();
			return -1;
		case 'V':
			/*@-mustfreefresh@ *//* splint and gettext, as above. */
			/* GNU standard first line format: program (package) and version only */
			printf("%s (%s) %s\n", common_program_name, PACKAGE_NAME, PACKAGE_VERSION);
			/* GNU standard second line format - "Copyright" always in English */
			printf("Copyright %s %s\n", COPYRIGHT_YEAR, COPYRIGHT_HOLDER);
			/* GNU standard license line and free software notice */
			printf("%s\n", _("License: GPLv3+ <https://www.gnu.org/licenses/gpl-3.0.html>"));
			printf("%s\n", _("This is free software: you are free to change and redistribute it."));
			printf("%s\n", _("There is NO WARRANTY, to the extent permitted by law."));
			/* Project web site link */
			printf("\n%s: <%s>\n", _("Project web site"), PACKAGE_URL);
			free_options();
			return -1;
			/*@+mustfreefresh@ */
		case 'c':
			if (parse_config(optarg, 0) != 0) {
				free_options();
				return 1;
			}
			config_specified = true;
			break;
		case 'D':
			pidfile = xstrdup(optarg);
			break;
		case 'd':
#if ENABLE_DEBUGGING
			debugging_enabled = true;
			break;
#else
			fprintf(stderr, "%s\n", _("Debugging is not enabled in this build."));
			free_options();
			return 1;
#endif
		default:
			/*@-formatconst@ */
			fprintf(stderr, _("Try `%s --help' for more information."), common_program_name);
			/*@+formatconst@ */
			/*
			 * splint note: formatconst is warning about the use
			 * of a non constant (translatable) format string;
			 * this is unavoidable here and the only attack
			 * vector is through the message catalogue.
			 */
			fprintf(stderr, "\n");
			free_options();
			return 1;
		}

	} while (c != -1);

	/*
	 * Read default config file if none was specified.
	 */
	if (!config_specified) {
		if (parse_config(DEFAULT_CONFIG_FILE, 0) != 0) {
			free_options();
			return 1;
		}
	}

	/*
	 * Store remaining command-line arguments.
	 */
	while (optind < argc) {
		config_sections_selected[config_sections_selected_count++]
		    = argv[optind++];
	}

	return 0;
}


/*
 * Become a daemon, i.e. detach from the controlling terminal and run in the
 * background.  Exits in the parent, returns in the child.
 */
static void daemonise(const char *pidfile)
{
	pid_t child;
	int fd;

	child = (pid_t) fork();

	if (child < 0) {
		/*
		 * Fork failed - abort.
		 */
		die("%s: %s", "fork", strerror(errno));
		exit(EXIT_FAILURE);
	}

	if (child > 0) {
		FILE *pidfptr;
		/*
		 * Fork succeeded and we're the parent process - write
		 * child's PID to the PID file and exit successfully.
		 */
		pidfptr = fopen(pidfile, "w");
		if (pidfptr == NULL) {
			create_parent_dirs(pidfile);
			pidfptr = fopen(pidfile, "w");
			if (pidfptr == NULL) {
				error("%s: %s", pidfile, strerror(errno));
				(void) kill(child, SIGTERM);
				exit(EXIT_FAILURE);
			}
		}
		fprintf(pidfptr, "%d\n", (int) child);
		(void) fclose(pidfptr);
		exit(EXIT_SUCCESS);
	}

	/*
	 * We're the background child process - cut our ties with the parent
	 * environment.
	 */
	fd = open("/dev/null", O_RDONLY);
	if (fd >= 0) {
		if (dup2(fd, 0) < 0)
			(void) close(0);
		(void) close(fd);
	} else {
		(void) close(0);
	}
	fd = open("/dev/null", O_WRONLY);
	if (fd >= 0) {
		if (dup2(fd, 1) < 0)
			(void) close(1);
		(void) close(fd);
	} else {
		(void) close(1);
	}
#if ENABLE_DEBUGGING
	if (!debugging_enabled) {
		if (dup2(1, 2) < 0)
			(void) close(2);
	}
#else				/* ENABLE_DEBUGGING */
	if (dup2(1, 2) < 0)
		(void) close(2);
#endif				/* ENABLE_DEBUGGING */

	(void) setsid();
}


/*
 * Handler for an exit signal such as SIGTERM - set a flag to trigger an
 * exit.
 */
static void sync_main_exitsignal( /*@unused@ */  __attribute__((unused))
				 int signum)
{
	sync_exit_now = true;
}


/*
 * Handler for a signal we do nothing with, such as SIGCHLD or SIGALRM.
 */
static void sync_main_nullsignal( /*@unused@ */  __attribute__((unused))
				 int signum)
{
	/* Do nothing. */
}


/*
 * Set up the signal handlers.
 */
static void set_signal_handlers(void)
{
	struct sigaction sa;

	memset(&sa, 0, sizeof(sa));

	sa.sa_handler = sync_main_exitsignal;
	(void) sigemptyset(&(sa.sa_mask));
	sa.sa_flags = 0;
	(void) sigaction(SIGTERM, &sa, NULL);

	sa.sa_handler = sync_main_exitsignal;
	(void) sigemptyset(&(sa.sa_mask));
	sa.sa_flags = 0;
	(void) sigaction(SIGINT, &sa, NULL);

	sa.sa_handler = sync_main_nullsignal;
	(void) sigemptyset(&(sa.sa_mask));
	sa.sa_flags = 0;
	(void) sigaction(SIGALRM, &sa, NULL);

	sa.sa_handler = sync_main_nullsignal;
	(void) sigemptyset(&(sa.sa_mask));
	sa.sa_flags = 0;
	(void) sigaction(SIGCHLD, &sa, NULL);
}


/*
 * Main entry point.  Read the command line arguments, parse the
 * configuration file(s), and start the main sync loop(s).
 */
int main(int argc, char **argv)
{
	int rc, sel_idx, cf_idx, defaults_idx;
	bool any_sections_chosen;
	char *env_path;
	bool rsync_null_delimiter;

#ifdef ENABLE_NLS
	/* Initialise language translation. */
	(void) setlocale(LC_ALL, "");
	(void) bindtextdomain(PACKAGE, LOCALEDIR);
	(void) textdomain(PACKAGE);
#endif

	/* Don't use argv[0], use canonical program name. */
	common_program_name = "continual-sync";

	rc = parse_options(argc, argv);
	if (rc < 0)
		return EXIT_SUCCESS;
	else if (rc > 0)
		return EXIT_FAILURE;

	/*
	 * Check we have some configuration sections.
	 */
	if (0 == config_sections_count) {
		error("%s", _("no configuration sections defined"));
		free_options();
		return EXIT_FAILURE;
	}

	/*
	 * Find the defaults section so we can use it later, and validate
	 * it.
	 */
	defaults_idx = find_config_section(DEFAULTS_SECTION);
	if (defaults_idx >= 0) {
		if (validate_config_section(defaults_idx, -1) != 0) {
			free_options();
			return EXIT_FAILURE;
		}
	}

	any_sections_chosen = false;

	/*
	 * Check that if we've chosen sections, they all exist and are
	 * valid, and mark them as selected.
	 */
	for (sel_idx = 0; sel_idx < config_sections_selected_count && NULL != config_sections_selected; sel_idx++) {
		cf_idx = find_config_section(config_sections_selected[sel_idx]);
		if (cf_idx < 0) {
			error("%s: %s", config_sections_selected[sel_idx], _("configuration section not found"));
			free_options();
			return EXIT_FAILURE;
		}
		if (strcmp(config_sections[cf_idx].name, DEFAULTS_SECTION)
		    == 0) {
			error("%s: %s", DEFAULTS_SECTION, _("this section cannot be used for synchronisation"));
			free_options();
			return EXIT_FAILURE;
		}
		if (validate_config_section(cf_idx, defaults_idx) != 0) {
			free_options();
			return EXIT_FAILURE;
		}
		config_sections[cf_idx].selected = true;
		any_sections_chosen = true;
	}

	/*
	 * If we chose no sections, we chose them all except
	 * DEFAULTS_SECTION, so check they all are valid in that case, and
	 * mark them all as selected.
	 */
	if (0 == config_sections_selected_count) {
		for (cf_idx = 0; cf_idx < config_sections_count; cf_idx++) {
			if (strcmp(config_sections[cf_idx].name, DEFAULTS_SECTION) == 0)
				continue;
			if (validate_config_section(cf_idx, defaults_idx)
			    != 0) {
				free_options();
				return EXIT_FAILURE;
			}
			config_sections[cf_idx].selected = true;
			any_sections_chosen = true;
		}
	}

	/*
	 * If there were no sections chosen, we cannot do anything.
	 */
	if (!any_sections_chosen) {
		error("%s", _("no sections were specified for synchronisation"));
		free_options();
		exit(EXIT_FAILURE);
	}

	/*
	 * Set a default PATH environment variable if we don't have one.
	 */
	env_path = getenv("PATH");
	if ((NULL == env_path) || ('\0' == env_path[0])) {
		/*@-unrecog@ */
		/* splint doesn't recognise putenv(). */
		putenv("PATH=/usr/bin:/bin:/usr/local/bin:/usr/sbin:/sbin:/usr/local/sbin");
		/*@+unrecog@ */
	}

	/*
	 * rsync has had "--from0" since 2.6.0 (2004); assume we can use it.
	 */
	rsync_null_delimiter = true;

	/*
	 * Become a daemon if necessary.
	 */
	if (NULL != pidfile) {
		daemonise(pidfile);
		/*@-unrecog@ */
		/* splint 3.1.2 chokes on syslog.h */
		openlog(common_program_name, LOG_PID, LOG_DAEMON);
		/*@+unrecog@ */
		using_syslog = true;
	}

	common_program_name = xstrdup(common_program_name);

	initproctitle(argc, argv);

	setproctitle("%s", common_program_name);

	/*
	 * Set up signal handling.
	 */
	set_signal_handlers();

	/*
	 * Write the section list file.
	 */
	if (NULL != config_sections[defaults_idx].section_list_file) {
		write_section_list_file(config_sections[defaults_idx].section_list_file, defaults_idx);
	}

	/*
	 * Main loop: maintain a child process for each selected section.
	 */
	while (!sync_exit_now) {
		/*
		 * Spawn any sync processes that need starting.
		 */
		for (cf_idx = 0; cf_idx < config_sections_count; cf_idx++) {
			pid_t child;

			if (!config_sections[cf_idx].selected)
				continue;
			if (config_sections[cf_idx].pid > 0)
				continue;

			child = (pid_t) fork();

			if (0 == child) {
				/* Child - run sync for this section */
				setproctitle("%s [%s]", common_program_name, config_sections[cf_idx].name);
				set_signal_handlers();
				continual_sync(&(config_sections[cf_idx]), rsync_null_delimiter);
				free_options();
				free(common_program_name);
				exit(EXIT_SUCCESS);
			} else if (child < 0) {
				/* Error - output a warning */
				error("%s: %s", "fork", strerror(errno));
			} else {
				/* Parent - store PID */
				config_sections[cf_idx].pid = child;
				debug("(master) pid %d spawned [%s]", child, config_sections[cf_idx].name);
			}
		}
		/*
		 * Clean up any child processes that have exited.
		 */
		for (cf_idx = 0; cf_idx < config_sections_count; cf_idx++) {
			if (!config_sections[cf_idx].selected)
				continue;
			if (config_sections[cf_idx].pid <= 0)
				continue;
			if (waitpid(config_sections[cf_idx].pid, NULL, WNOHANG) != 0) {
				debug("(master) pid %d exited [%s]",
				      config_sections[cf_idx].pid, config_sections[cf_idx].name);
				config_sections[cf_idx].pid = 0;
			}
		}
		(void) usleep(100000);
	}

	/*
	 * Kill any still-running sync processes.
	 */
	for (cf_idx = 0; cf_idx < config_sections_count; cf_idx++) {
		if (!config_sections[cf_idx].selected)
			continue;
		if (config_sections[cf_idx].pid <= 0)
			continue;
		(void) kill(config_sections[cf_idx].pid, SIGTERM);
	}

	if (NULL != pidfile) {
		(void) remove(pidfile);
		closelog();
	}
	free_options();
	free(common_program_name);

	return EXIT_SUCCESS;
}
