#!/bin/sh
#
# Function to launch the test subject under valgrind.
#
# If valgrind is unavailable, exits the script with status 77, after writing
# a note to file descriptor 9.
#
# If valgrind finds an error, writes the error to "valgrind.out" in the
# current directory, and exits the script with status 1 after writing a note
# to file descriptor 9.
#
# If valgrind does not find any errors, the function returns with the exit
# status of the test subject.
#
# Source this file from test scripts that use valgrind.
#
# Requires ${testSubject} and ${workDir}.
#

# Output file for failures.
valgrindOutputFile="valgrind.out"

true "${testSubject:?not set - call this from 'make check'}"
true "${workDir:?not set - call this from 'make check'}"

if ! command -v valgrind >/dev/null 2>&1; then
	printf "%s\n" "test requires \`valgrind'" 1>&9
	exit 77
fi

if test "${SKIP_VALGRIND_TESTS}" = "1"; then
	echo "SKIP_VALGRIND_TESTS is set"
	exit 77
fi

valgrindHelp="$(valgrind --help 2>&1)"
for valgrindOption in "verbose" "show-error-list" "error-exitcode" "track-fds" "leak-check"; do
	printf "%s\n" "${valgrindHelp}" | grep -Fq "${valgrindOption}" || { printf "%s\n" "test requires \`valgrind --${valgrindOption}'" 1>&9; exit 77; }
done

runWithValgrind () {

	valgrind --tool=memcheck \
	  --verbose --show-error-list=yes --log-file="${workDir}/valgrind.out" \
	  --error-exitcode=125 \
	  --track-fds=yes \
	  --leak-check=full \
	  "${testSubject}" "$@" \
	  9<&-

	returnValue=$?

	cat "${workDir}/valgrind.out"

	if test "${returnValue}" -eq 125; then
		{
		printf "%s\n" "================================================"
		date
		printf "%s\n" "Command: ${testSubject} $*"
		echo
		cat "${workDir}/valgrind.out"
		printf "%s\n" "================================================"
		echo
		} >> "${valgrindOutputFile}"
		printf "%s\n" "memory check failed - see file \`valgrind.out'." 1>&9
		exit 1
	fi

	return "${returnValue}"
}
