#!/bin/sh
#
# Functions for maintaining a container image for building Ubuntu packages.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Ensure that the container image "${buildImageName}" exists.
#
build_image_ubuntu () {
	# Do nothing if the image already exists.
	dockerCommand image inspect "${buildImageName}" >/dev/null 2>&1 && return 0

	case "${targetOs}" in
	'ubuntu2204') osVersion='22.04' ;;
	'ubuntu2404') osVersion='24.04' ;;
	'ubuntu2504') osVersion='25.04' ;;
	'ubuntu2510') osVersion='25.10' ;;
	'ubuntu2604') osVersion='26.04' ;;
	*) reportError "${buildImageName}: unrecognised OS version"; return "${RC_BAD_ARGS}" ;;
	esac

	# Write the build instructions.
	{
	printf '%s%s\n' 'FROM ubuntu:' "${osVersion}"
	printf '%s\n' 'ADD packageList /tmp/packageList'
	# shellcheck disable=SC2016
	printf '%s %s %s\n' \
	  'RUN apt-get update && DEBIAN_FRONTEND=noninteractive apt-get -y upgrade' \
	  '&& DEBIAN_FRONTEND=noninteractive apt-get -y install $(cat /tmp/packageList)' \
	  '&& DEBIAN_FRONTEND=noninteractive apt-get -y autoremove && apt-get clean && rm -f /tmp/packageList'
	} > "${workDir}/Dockerfile"
	{
	# Can't build much without these:
	printf '%s\n' 'build-essential debhelper devscripts'
	# Need these for maintaining repositories:
	printf '%s\n' 'apt-utils'
	# Fallback needed if signing required:
	printf '%s\n' 'pinentry-tty'
	# Needed to answer passphrase prompts:
	printf '%s\n' 'expect'
	# Commonly needed packages:
	printf '%s\n' 'cmake pkg-config default-jdk'
	printf '%s\n' 'libcurl4-openssl-dev libffi-dev libkrb5-dev libldap2-dev librtmp-dev libssh2-1-dev libssl-dev'
	printf '%s\n' 'libjson-c5 libjson-c-dev'
	printf '%s\n' 'libidn-dev'
	} > "${workDir}/packageList"

	reportProgress 'building the base image'

	# Build the image.
	(
	cd "${workDir}" || exit "${RC_LOCAL_FAULT}"
	dockerCommand build --tag "${buildImageName}" . || exit "${RC_LOCAL_FAULT}"
	exit 0
	)

	# Check that the image built.
	dockerCommand image inspect "${buildImageName}" >/dev/null 2>&1 && return 0

	reportError "${buildImageName}: failed to create base image"
	return "${RC_LOCAL_FAULT}"
}


# Apply updates to the container image "${buildImageName}".
#
update_image_ubuntu () {
	build_image_ubuntu || return $?
	updateBuildImage 'export DEBIAN_FRONTEND=noninteractive; apt-get update; apt-get -y upgrade; apt-get -y autoremove; apt-get clean'
}


# Install packages into the container image "${buildImageName}".
#
install_into_image_ubuntu () {
	build_image_ubuntu || return $?
	updateBuildImage 'apt-get update; apt-get -y install '"$(printf '%s\n' "$*" | tr -dc ' \t0-9A-Za-z_.-')"' || exit $?; apt-get clean'
}
