#!/bin/sh
#
# Functions for generating build instructions.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Read man page format on stdin and produce plain text on stdout.
#
manToText () {
	groff -mandoc -Tutf8 -a | sed 's,<\(\hy\|-\)>,-,g;s,<.*>,,g'
}

# Generate build instructions for the source tarball at ${sourcePath}, using
# ${workDir} as scratch space, and write them to the directory $1.
#
# If running inside a container, sourcePath is set to $2.
#
generateBuildInstructions () {
	instructionsDirectory="$1"
	test "${action}" = 'inside-container' && sourcePath="$2"

	if ! test "${action}" = 'inside-container'; then
		test -e "${workDir}/extracted" && rm -rf "${workDir}/extracted"
		mkdir -p "${workDir}/extracted"
		extractSource "${workDir}/extracted" || return $?
	fi

	mkdir -p "${workDir}/instructions"

	# In container mode, do the rest inside a container instead.
	if ! ${nativeOnly}; then
		if ! test "${action}" = 'inside-container'; then
			loadComponent 'containers'
			runInContainer 'containers build-instructions' 'build-instructions' '/mnt/instructions' "${sourcePath}" || return $?
			# Copy the generated instructions to the destination.
			cp "${workDir}/instructions/"* "${instructionsDirectory}/" || return "${RC_LOCAL_FAULT}"
			return 0
		fi
	fi

	useName=""		# package name
	useVersion=""		# package version
	useSummary=""		# one-line summary of the package
	useDescription=""	# multi-line description of the package

	# Attempt to derive the name and version from the top-level
	# directory in the source archive.
	useName="$(find "${workDir}/extracted" -mindepth 1 -maxdepth 1 -type d -name '[A-Za-z]*-[0-9]*.*' -printf '%f\n' \
	           | sed -n '/-\([0-9.][0-9a-z.]*\)$/p' | sed -n 1p)"
	if test -n "${useName}"; then
		useVersion="${useName##*-}"
		useName="${useName%-*}"
	fi

	# Configure and build the source, and install to a temporary
	# location - this means that the source must honour "DESTDIR" in
	# "make install".
	if test -e "${workDir}/extracted/configure"; then
		(
		cd "${workDir}/extracted" \
		&& sh ./configure \
		     --prefix=/usr \
		     --exec-prefix=/usr \
		     --bindir=/usr/bin \
		     --sbindir=/usr/sbin \
		     --sysconfdir=/etc \
		     --datadir=/usr/share \
		     --includedir=/usr/include \
		     --libexecdir=/usr/libexec \
		     --localstatedir=/var \
		     --sharedstatedir=/var/lib \
		     --mandir=/usr/share/man \
		     --infodir=/usr/share/info
		) || return "${RC_LOCAL_FAULT}"
	fi
	make -C "${workDir}/extracted" || return "${RC_LOCAL_FAULT}"
	mkdir -p "${workDir}/trialInstall"
	make -C "${workDir}/extracted" install DESTDIR="${workDir}/trialInstall" || return "${RC_LOCAL_FAULT}"

	# Check that anything was installed.
	if find "${workDir}/trialInstall" -type f | grep -Ec . | grep -Fqx '0'; then
		reportError "${sourcePath}: trial installation did not install any files under DESTDIR"
		rm -rf "${workDir}/trialInstall"
		return "${RC_LOCAL_FAULT}"
	fi

	# Count how many binary executables were installed, so that the
	# build instructions can say whether or not this package is
	# architecture-dependent.
	executableInstallCount="$(
	  LC_ALL=C find "${workDir}/trialInstall" -type f -exec file --brief '{}' ';' \
	  | grep -Fvw 'text' \
	  | grep -Fwc 'executable'
	)"

	# Look in the software's installed manual pages to find a summary
	# and description (and package name, if one was not found yet).
	find "${workDir}/trialInstall/usr/share/man" "${workDir}/extracted" -type f -name "*.[0-9]" > "${workDir}/manuals" 2>/dev/null
	{
	while read -r manualFile; do
		test -n "${manualFile}" || continue

		# Read the first non-blank line after the start of the NAME
		# section; it should be of the form "program - summary".
		grep -Ev '^[[:space:]]*$' "${manualFile}" \
		| grep -E -A 1 '^\.SH +NAME *$' \
		| sed -n '$p' \
		| manToText \
		| grep -E . \
		> "${workDir}/manualNameLine"

		# Skip this manual if the line wasn't there.
		test -s "${workDir}/manualNameLine" || continue

		# Skip if the line wasn't in the right format.
		read -r newName separator newSummary < "${workDir}/manualNameLine"
		test -n "${newName}" || continue
		test -n "${separator}" || continue
		test -n "${newSummary}" || continue

		# Skip if we think we know the package name and this manual
		# doesn't have the same name.
		if test -n "${useName}"; then
			test "${useName}" = "${newName}" || continue
		fi

		# Accept the manual's idea of the package name and summary;
		# capitalise the first letter of the summary.
		useName="${newName}"
		useSummary="$(printf '%s\n' "${newSummary}" | sed 's/^\(.\).*$/\1/' | tr '[:lower:]' '[:upper:]')${newSummary#?}"

		# Find the line number in the manual just before the first
		# paragraph or section break occurs after the start of the
		# "DESCRIPTION" section, so we can use the first paragraph
		# as the package description.
		lastLineNumberOfDescription="$(
		  sed -n '/^\.SH \+DESCRIPTION/,/^\(\.[TPS]\|$\)/=' < "${manualFile}" \
		  | tac \
		  | sed -n 2p
		)"

		if test -n "${lastLineNumberOfDescription}"; then
			# Reflow the text of the paragraph so that words
			# that were split on a hyphen are rejoined and the
			# paragraph width is 77 characters, as the default
			# format gets a bit enthusiastic about splitting
			# words.
			newDescription="$(
			  sed -n "1,${lastLineNumberOfDescription}p" < "${manualFile}" \
			  | manToText \
			  | sed -e '1,/^DESCRIPTION/d' -e '$d' -e 's/^ //' \
			  | tr '\n' '\r' \
			  | sed 's/-\r//g' \
			  | tr '\r' '\n' \
			  | fmt -w 77 \
			  | grep -E .
			)"
			test -n "${newDescription}" && useDescription="${newDescription}"
		fi
	done
	} < "${workDir}/manuals"

	# If the version is not yet known, look for any "NEWS" or
	# "changelog" files and use the first version number found.
	if test -z "${useVersion}"; then
		find "${workDir}/trialInstall/usr/share/doc" "${workDir}/extracted" \
		  -type f \( -name "NEWS*" -o -iname "changelog" \) \
		  > "${workDir}/changelogs" 2>/dev/null
		{
		while read -r changelogFile; do
			test -n "${changelogFile}" || continue
			newVersion="$(
			  sed -n 's/\(^\|[^0-9]\+\)\([0-9][0-9.]*\)\([^0-9.].*\|$\)/ \2 /p' < "${changelogFile}" \
			  | awk '{print $1}' \
			  | sed -n 1p
			)"
			test -n "${newVersion}" || continue
			useVersion="${newVersion}"
			break
		done
		} < "${workDir}/changelogs"
	fi

	# Check we have the bare minimum to be able to proceed.
	test -n "${useName}" || { reportError "${sourcePath}: unable to determine package name"; rm -rf "${workDir}/trialInstall"; return "${RC_BAD_ARGS}"; }
	test -n "${useVersion}" || { reportError "${sourcePath}: unable to determine package version"; rm -rf "${workDir}/trialInstall"; return "${RC_BAD_ARGS}"; }
	test -n "${useSummary}" || useSummary='Autogenerated package'
	test -n "${useDescription}" || useDescription='This package was automatically generated from sources.'

	# Write the RPM spec file.

	{
	printf 'Summary: %s\n' "${useSummary}"
	printf 'Name: %s\n' "${useName}"
	printf 'Version: %s\n' "${useVersion}"
	printf 'Release: %s\n' '1%{?dist}'
	printf 'License: %s\n' 'unknown'
	printf 'Source: %s\n' "${sourcePath##*/}"
	test "${executableInstallCount}" -gt 0 || printf 'BuildArch: noarch\n'
	printf '\n'
	printf '%s\n' '%description'
	printf '%s\n' "${useDescription}"
	printf '\n'
	printf '%s\n' '%prep'
	printf '%s\n' '%setup -q'
	printf '\n'
	printf '%s\n' '%build'
	test -e "${workDir}/extracted/configure" && printf '%s\n' '%configure'
	printf '%s\n' 'make %{?_smp_mflags}'
	printf '\n'
	printf '%s\n' '%install'
	# shellcheck disable=SC2016
	{
	printf '%s\n' 'test -n "${RPM_BUILD_ROOT}" && test "${RPM_BUILD_ROOT}" != "/" && rm -rf "${RPM_BUILD_ROOT}"'
	printf '%s\n' 'make install DESTDIR="${RPM_BUILD_ROOT}"'
	printf '%s\n' 'rm -rf "${RPM_BUILD_ROOT}/usr/share/doc"'
	printf '\n'
	printf '%s\n' '%clean'
	printf '%s\n' 'test -n "${RPM_BUILD_ROOT}" && test "${RPM_BUILD_ROOT}" != "/" && rm -rf "${RPM_BUILD_ROOT}"'
	}
	printf '\n'
	printf '%s\n' '%files'
	printf '%s\n' '%defattr(-, root, root)'
	find "${workDir}/trialInstall" -type f -printf '/%P\n' | grep -Fv '/usr/share/doc' > "${workDir}/installedFiles"
	{
	while read -r installedFile; do
		case "${installedFile}" in
		/etc/*)		printf '%s %s\n' '%config(noreplace)' "${installedFile}" ;;
		*/man/*)	printf '%s*\n' "${installedFile}" ;;
		*)		printf '%s\n' "${installedFile}" ;;
		esac
	done
	} < "${workDir}/installedFiles"
	find "${workDir}/trialInstall" -mindepth 2 -type d -printf '/%P\n' | grep -Fv '/usr/share/doc' > "${workDir}/installedDirectories"
	{
	while read -r installedDir; do
		test -n "${installedDir}" || continue
		# Skip if this directory has subdirectories.
		grep -qF "${installedDir}/" < "${workDir}/installedDirectories" && continue
		# Skip if this directory contains files.
		grep -qF "${installedDir}/" < "${workDir}/installedFiles" && continue
		printf '%s %s\n' '%dir' "${installedDir}"
	done
	} < "${workDir}/installedDirectories"
	printf '\n'
	# The "%doc" line needs to be one line listing the source files that
	# are to be installed in /usr/share/doc.
	find "${workDir}/trialInstall/usr/share/doc" -type f -printf '%f\n' > "${workDir}/installedDocs" 2>/dev/null
	if test -s "${workDir}/installedDocs"; then
		{
		while read -r docFile; do
			# To avoid getting "pkg-0.1.2/doc/FILE" when we want
			# "doc/FILE", look under "pkg-*".
			find "${workDir}/extracted/${useName}"*/ -type f -name "${docFile}" -printf '%P\n' 2>/dev/null \
			| sed -n 1p
		done
		} < "${workDir}/installedDocs" > "${workDir}/docSourceList"
		if test -s "${workDir}/docSourceList"; then
			printf '%s %s\n' '%doc' "$(tr '\n' ' ' < "${workDir}/docSourceList" | sed 's/ $//')"
			printf '\n'
		fi
	fi
	# Dummy changelog.
	printf '%s\n' '%changelog'
	printf '* %s %s <> - %s-1\n' "$(date '+%a %b %d %Y')" "${PACKAGE_NAME}" "${useVersion}"
	printf '%s %s\n' '-' 'Build instructions generated from source.'
	} > "${workDir}/instructions/rpm.spec"
	
	# Write the Debian build files.

	# Debian contrrol file.
	{
	printf 'Source: %s\n' "${useName}"
	printf 'Maintainer: %s <>\n' "${PACKAGE_NAME}"
	printf 'Section: %s\n' 'misc'
	printf 'Priority: %s\n' 'optional'
	printf 'Standards-Version: %s\n' '3.9.2'
	printf 'Build-Depends: debhelper (>= 9)\n'
	printf '\n'
	printf 'Package: %s\n' "${useName}"
	if test "${executableInstallCount}" -gt 0; then
		printf 'Architecture: %s\n' 'any'
	else
		printf 'Architecture: %s\n' 'all'
	fi
	# shellcheck disable=SC2016
	printf 'Depends: %s\n' '${misc:Depends}'
	printf 'Description: %s\n' "${useSummary}"
	printf '%s\n.\n' "${useDescription}" | sed 's,^, ,'
	} > "${workDir}/instructions/control"

	# Boilerplate rules file.
	{
	printf '%s\n' '#!/usr/bin/make -f'
	printf '\n' 
	printf '%s\n' 'export DH_VERBOSE = 1'
	printf '%s\n' 'export DEB_BUILD_MAINT_OPTIONS = hardening=+all'
	printf '%s\n' 'export DEB_CFLAGS_MAINT_APPEND  = -Wall -pedantic'
	printf '%s\n' 'export DEB_LDFLAGS_MAINT_APPEND = -Wl,--as-needed'
	printf '\n'
	printf '%s\n' '%:'
	printf '\t%s\n' 'dh $@'
	} > "${workDir}/instructions/rules"
	chmod 755 "${workDir}/instructions/rules"

	# Dummy changelog.
	{
	printf '%s (%s-1) %s\n' "${useName}" "${useVersion}" 'UNRELEASED; urgency=low'
	printf '\n'
	printf '  * %s\n' 'Build instructions generated from source.'
	printf '\n'
	printf ' -- %s <>  %s\n' "${PACKAGE_NAME}" "$(date -R)"
	} > "${workDir}/instructions/changelog"

	# Remove the trial installation, to avoid permissions problems
	# outside the container when running under Docker.
	rm -rf "${workDir}/trialInstall"

	# Remove the extracted source, for the same reason, as some builds
	# modify the source tree with things like ".deps" directories.
	rm -rf "${workDir}/extracted"

	# Copy the generated instructions to the destination.
	if ! test "${action}" = 'inside-container'; then
		cp "${workDir}/instructions/"* "${instructionsDirectory}/" || return "${RC_LOCAL_FAULT}"
	fi

	return 0
}
