#!/bin/sh
#
# Functions for building CentOS 7 packages.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Build packages for CentOS 7 from the extracted source at
# ${workDir}/build-source/ and place them in ${destinationPath}, using the
# build instructions at ${workDir}/build-instructions/.
#
# If $1 is "check-prerequisites", just check that the commands are available
# to build packages for the target OS, returning 0 if OK,
# ${RC_NOT_SUPPORTED} if not OK.
#
build_package_centos7 () {
	if test "$1" = 'check-prerequisites'; then
		for checkCommand in 'rpm' 'rpmbuild'; do
			command -v "${checkCommand}" >/dev/null 2>&1 && continue
			reportError "${checkCommand}: command not found - cannot build packages for ${targetOs}"
			return "${RC_NOT_SUPPORTED}"
		done
		return 0
	fi

	rm -rf "${workDir}/packages"
	mkdir -p "${workDir}/packages"

	# In container mode, do the rest inside a container instead.
	if ! ${nativeOnly}; then
		if ! test "${action}" = 'inside-container'; then
			loadComponent 'containers'
			runInContainer 'containers build-package-centos7' 'build-package' || return $?
			# Copy the generated packages to the destination.
			cp "${workDir}/packages/"* "${destinationPath}/" || return "${RC_LOCAL_FAULT}"
			return 0
		fi
	fi

	# Set up the GPG keyring and $signingKey if appropriate.
	setupSigningKey || return $?

	# Set up a build directory with a copy of the source and the
	# instructions.  This ensures that when running in a container, the
	# interaction with the files outside the container is minimised,
	# which avoids (with some container types) dropping too many
	# root-owned files around.
	#
	buildDir="${workDir}/build"
	test "${action}" = 'inside-container' && buildDir="/root/${PACKAGE_NAME}"
	mkdir -p "${buildDir}/rpmbuild"
	for subdir in 'BUILD' 'BUILDROOT' 'RPMS' 'SOURCES' 'SPECS' 'SRPMS'; do
		mkdir -p "${buildDir}/rpmbuild/${subdir}"
	done
	tar cf - -C "${workDir}/build-source" . | tar xf - -C "${buildDir}/rpmbuild/SOURCES" || return "${RC_LOCAL_FAULT}"
	tar cf - -C "${workDir}/build-instructions" . | tar xf - -C "${buildDir}/rpmbuild/SPECS" || return "${RC_LOCAL_FAULT}"

	# Build the package.
	rpmbuild -D "_topdir ${buildDir}/rpmbuild" -bb "${buildDir}/rpmbuild/SPECS/"*spec || return "${RC_LOCAL_FAULT}"

	# Check whether the RPMs were placed in the rpmbuild directory under
	# RPMS/<arch>/, as normal, or whether they are under RPMS/ (the
	# MariaDB 10.5.26 source RPM does this).
	rpmSubdirDepth='2'
	if find "${buildDir}/rpmbuild/RPMS" -mindepth 1 -maxdepth 1 -type f -name '*.rpm' | grep -q . \
	   && ! find "${buildDir}/rpmbuild/RPMS" -mindepth 2 -maxdepth 2 -type f -name '*.rpm' | grep -q .; then
		rpmSubdirDepth='1'
	fi

	if test -n "${signingKey}"; then
		if test "${rpmSubdirDepth}" -eq 2; then
			answerPassphrasePrompts rpm -D "%_gpg_name ${signingKey}" --addsign "${buildDir}/rpmbuild/RPMS"/*/*rpm \
			|| return "${RC_LOCAL_FAULT}"
		else
			answerPassphrasePrompts rpm -D "%_gpg_name ${signingKey}" --addsign "${buildDir}/rpmbuild/RPMS"/*rpm \
			|| return "${RC_LOCAL_FAULT}"
		fi
	fi

	# Generate summary files for each package, and copy the packages and
	# their summary files to the work directory.
	find "${buildDir}/rpmbuild/RPMS" -mindepth "${rpmSubdirDepth}" -maxdepth "${rpmSubdirDepth}" -type f -name '*.rpm' \
	| while read -r packagePath; do
		test -e "${packagePath}" || continue
		packageFile="${packagePath##*/}"
		cp "${packagePath}" "${workDir}/packages/${packageFile}" || return "${RC_LOCAL_FAULT}"
		{
		printf '%s: %s\n' 'OS' "${targetOs}"
		# shellcheck disable=SC2016
		rpm --query \
		  --queryformat='Name: %{NAME}\nVersion: %{VERSION}-%{RELEASE}\nArchitecture: %{ARCH}\nSummary: %{SUMMARY}\n %{DESCRIPTION}\n' \
		  --package "${packagePath}"
		} \
		| sed '/^ /,$s/^/ /;s/^  / /' \
		> "${workDir}/packages/${packageFile}.txt"
	done

	# Copy the generated packages to the destination.
	if ! test "${action}" = 'inside-container'; then
		cp "${workDir}/packages/"* "${destinationPath}/" || return "${RC_LOCAL_FAULT}"
	fi

	return 0
}


# Extract build instructions for this OS from the extracted source under
# ${workDir}/build-source/ and place them under
# ${workDir}/build-instructions/ if any were found.
#
extract_instructions_centos7 () {
	extractInstructionsForRPM "${workDir}/build-source" "${workDir}/build-instructions"
}
