#!/bin/sh
#
# Functions for building Debian packages.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Build packages for Debian from the extracted source at
# ${workDir}/build-source/ and place them in ${destinationPath}, using the
# build instructions at ${workDir}/build-instructions/.
#
# If $1 is "check-prerequisites", just check that the commands are available
# to build packages for the target OS, returning 0 if OK,
# ${RC_NOT_SUPPORTED} if not OK.
#
build_package_debian () {
	if test "$1" = 'check-prerequisites'; then
		for checkCommand in 'dpkg-deb' 'dpkg-buildpackage'; do
			command -v "${checkCommand}" >/dev/null 2>&1 && continue
			reportError "${checkCommand}: command not found - cannot build packages for ${targetOs}"
			return "${RC_NOT_SUPPORTED}"
		done
		return 0
	fi

	rm -rf "${workDir}/packages"
	mkdir -p "${workDir}/packages"

	# In container mode, do the rest inside a container instead.
	# NB we bring in the Ubuntu build functions too, since the Ubuntu
	# build components use the Debian ones.
	if ! ${nativeOnly}; then
		if ! test "${action}" = 'inside-container'; then
			loadComponent 'containers'
			runInContainer 'containers build-package-debian build-package-ubuntu' 'build-package' || return $?
			# Copy the generated packages to the destination.
			cp "${workDir}/packages/"* "${destinationPath}/" || return "${RC_LOCAL_FAULT}"
			return 0
		fi
	fi

	# Set up the GPG keyring and $signingKey if appropriate.
	setupSigningKey || return $?

	# Set up a build directory with a copy of the source and the
	# instructions.  This ensures that when running in a container, the
	# interaction with the files outside the container is minimised,
	# which avoids (with some container types) dropping too many
	# root-owned files around.
	#
	# The directory needs to be two levels down because
	# dpkg-buildpackage places the packages one directory up from the
	# source.
	#
	buildDir="${workDir}/build"
	test "${action}" = 'inside-container' && buildDir="/root/${PACKAGE_NAME}"
	mkdir -p "${buildDir}/source"
	tar cf - -C "${workDir}/build-source" . | tar xf - -C "${buildDir}/source" || return "${RC_LOCAL_FAULT}"
	mkdir -p "${buildDir}/source/debian"
	tar cf - -C "${workDir}/build-instructions" . | tar xf - -C "${buildDir}/source/debian" || return "${RC_LOCAL_FAULT}"

	# Add some defaults if certain build instructions files are missing.
	mkdir -p "${buildDir}/source/debian/source"
	test -e "${buildDir}/source/debian/source/format" || printf "%s\n" "3.0 (quilt)" > "${buildDir}/source/debian/source/format"
	test -e "${buildDir}/source/debian/compat" || printf "%s\n" "10" > "${buildDir}/source/debian/compat"
	test -e "${buildDir}/source/debian/copyright" || touch "${buildDir}/source/debian/copyright"
	chmod 700 "${buildDir}/source/debian/rules"

	# Build the package.
	(
	if test -n "${signingKey}"; then
		cd "${buildDir}/source" && answerPassphrasePrompts dpkg-buildpackage --build=binary --force-sign --sign-key="${signingKey}"
		rc=$?
	else
		cd "${buildDir}/source" && dpkg-buildpackage --build=binary
		rc=$?
	fi
	test "${rc}" -eq 0
	) || return "${RC_LOCAL_FAULT}"

	# Generate summary files for each package, and copy the packages and
	# their summary files to the work directory.
	find "${buildDir}" -mindepth 1 -maxdepth 1 -type f -name '*.deb' -printf '%f\n' \
	| while read -r packageFile; do
		test -e "${buildDir}/${packageFile}" || continue
		cp "${buildDir}/${packageFile}" "${workDir}/packages/${packageFile}" || return "${RC_LOCAL_FAULT}"
		for suffix in 'buildinfo' 'changes'; do
			test -e "${buildDir}/${packageFile%.deb}.${suffix}" || continue
			cp "${buildDir}/${packageFile%.deb}.${suffix}" "${workDir}/packages/${packageFile%.deb}.${suffix}" || return "${RC_LOCAL_FAULT}"
		done
		{
		printf '%s: %s\n' 'OS' "${targetOs}"
		# shellcheck disable=SC2016
		dpkg-deb --show \
		  --showformat='Name: ${Package}\nVersion: ${Version}\nArchitecture: ${Architecture}\nSummary: ${Description}\n' \
		  "${buildDir}/${packageFile}"
		} > "${workDir}/packages/${packageFile}.txt"
	done

	# Copy the generated packages to the destination.
	if ! test "${action}" = 'inside-container'; then
		cp "${workDir}/packages/"* "${destinationPath}/" || return "${RC_LOCAL_FAULT}"
	fi

	return 0
}


# Extract build instructions for this OS from the extracted source under
# ${workDir}/build-source/ and place them under
# ${workDir}/build-instructions/ if any were found.
#
extract_instructions_debian () {
	extractInstructionsForDebian "${workDir}/build-source" "${workDir}/build-instructions"
}
