#!/bin/sh
#
# Functions for indexing repositories for AlmaLinux packages.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Index the CentOS 7 packages in directory $1, and write the index to
# directory $2.  The two directories can be the same.
#
# If $3 is "check-prerequisites", just check that the commands are available
# to index repositories for the target OS, returning 0 if OK,
# ${RC_NOT_SUPPORTED} if not OK.
#
index_repository_almalinux () {
	repoSourceDir="$1"
	repoIndexDir="$2"

	if test "$3" = 'check-prerequisites'; then
		# shellcheck disable=SC2041
		for checkCommand in 'createrepo'; do
			command -v "${checkCommand}" >/dev/null 2>&1 && continue
			reportError "${checkCommand}: command not found - cannot index repositories for ${targetOs}"
			return "${RC_NOT_SUPPORTED}"
		done
		return 0
	fi

	# In container mode, do the rest inside a container instead.
	if ! ${nativeOnly}; then
		if ! test "${action}" = 'inside-container'; then
			loadComponent 'containers'
			runInContainer \
			  "volume:$(realpath "${repoSourceDir}"):/mnt/packages:ro" \
			  'containers index-repository-almalinux index-repository-centos7' \
			  'index-repository' '/mnt/packages' "${repoIndexDir}" \
			|| return $?
			type index_repository_centos7 >/dev/null 2>&1 || loadComponent 'index-repository-centos7'
			index_repository_replace_centos7 "${repoIndexDir}" || return $?
			return 0
		fi
	fi

	# The process is the same as for CentOS 7.
	type index_repository_centos7 >/dev/null 2>&1 || loadComponent 'index-repository-centos7'
	index_repository_centos7 "$@"
	return $?
}
