#!/bin/sh
#
# Functions for indexing repositories for CentOS 7 packages.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Replace the repository index in directory $1 with the one that is in
# ${workDir}/index.
#
index_repository_replace_centos7 () {
	repoIndexDir="$1"

	for subDir in 'cache' 'drpms' 'repodata'; do
		test -d "${workDir}/index/${subDir}" || continue

		rm -rf "${repoIndexDir}/.${subDir}.new"
		cp -dR "${workDir}/index/${subDir}" "${repoIndexDir}/.${subDir}.new" || return "${RC_LOCAL_FAULT}"

		chmod -R a+rX "${repoIndexDir}/.${subDir}.new"

		if test -e "${repoIndexDir}/${subDir}"; then
			rm -rf "${repoIndexDir}/.${subDir}.old"
			mv "${repoIndexDir}/${subDir}" "${repoIndexDir}/.${subDir}.old" || return "${RC_LOCAL_FAULT}"
			mv "${repoIndexDir}/.${subDir}.new" "${repoIndexDir}/${subDir}" || return "${RC_LOCAL_FAULT}"
			rm -rf "${repoIndexDir}/.${subDir}.old"
		else
			mv "${repoIndexDir}/.${subDir}.new" "${repoIndexDir}/${subDir}" || return "${RC_LOCAL_FAULT}"
		fi
	done

	return 0
}

# Index the CentOS 7 packages in directory $1, and write the index to
# directory $2.  The two directories can be the same.
#
# If $3 is "check-prerequisites", just check that the commands are available
# to index repositories for the target OS, returning 0 if OK,
# ${RC_NOT_SUPPORTED} if not OK.
#
index_repository_centos7 () {
	repoSourceDir="$1"
	repoIndexDir="$2"

	if test "$3" = 'check-prerequisites'; then
		# shellcheck disable=SC2041
		for checkCommand in 'createrepo'; do
			command -v "${checkCommand}" >/dev/null 2>&1 && continue
			reportError "${checkCommand}: command not found - cannot index repositories for ${targetOs}"
			return "${RC_NOT_SUPPORTED}"
		done
		return 0
	fi

	rm -rf "${workDir}/index"
	mkdir -p "${workDir}/index"
	test -d "${repoIndexDir}/cache" && cp -a "${repoIndexDir}/cache" "${workDir}/index/cache"

	# In container mode, do the rest inside a container instead.
	if ! ${nativeOnly}; then
		if ! test "${action}" = 'inside-container'; then
			loadComponent 'containers'
			runInContainer \
			  "volume:$(realpath "${repoSourceDir}"):/mnt/packages:ro" \
			  'containers index-repository-centos7' \
			  'index-repository' '/mnt/packages' "${repoIndexDir}" \
			|| return $?
			index_repository_replace_centos7 "${repoIndexDir}" || return $?
			return 0
		fi
	fi

	# Use a subshell since we will be using "cd".
	(
	set -e
	cd "${repoSourceDir}"
	# NB AlmaLinux 9's createrepo doesn't have "--deltas".
	if createrepo --help 2>&1 | grep -Fq deltas; then
		createrepo --update --deltas -c cache --outputdir "${workDir}/index" .
	else
		createrepo --update -c cache --outputdir "${workDir}/index" .
	fi
	) || return "${RC_LOCAL_FAULT}"

	# Copy the generated packages to the destination.
	if ! test "${action}" = 'inside-container'; then
		index_repository_replace_centos7 "${repoIndexDir}" || return $?
	fi

	# Inside a container, ensure global write permission so the outside
	# can remove the files and directories we created.
	test "${action}" = 'inside-container' && chmod -R a+rwX "${workDir}/index"

	return 0
}
