#!/bin/sh
#
# Functions for indexing repositories for Debian packages.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Replace the repository index in directory $1 with the one that is in
# ${workDir}/index.
#
index_repository_replace_debian () {
	repoIndexDir="$1"

	rm -rf "${repoIndexDir}/.dists.new"
	mkdir "${repoIndexDir}/.dists.new" || return "${RC_LOCAL_FAULT}"

	cp -dR "${workDir}/index/"* "${repoIndexDir}/.dists.new/" || return "${RC_LOCAL_FAULT}"

	chmod -R a+rX "${repoIndexDir}/.dists.new"

	if test -e "${repoIndexDir}/dists"; then
		rm -rf "${repoIndexDir}/.dists.old"
		mv "${repoIndexDir}/dists" "${repoIndexDir}/.dists.old" || return "${RC_LOCAL_FAULT}"
		mv "${repoIndexDir}/.dists.new" "${repoIndexDir}/dists" || return "${RC_LOCAL_FAULT}"
		rm -rf "${repoIndexDir}/.dists.old"
	else
		mv "${repoIndexDir}/.dists.new" "${repoIndexDir}/dists" || return "${RC_LOCAL_FAULT}"
	fi

	return 0
}

# Index the Debian packages in directory $1, and write the index to
# directory $2.  The two directories can be the same.
#
# If $3 is "check-prerequisites", just check that the commands are available
# to index repositories for the target OS, returning 0 if OK,
# ${RC_NOT_SUPPORTED} if not OK.
#
index_repository_debian () {
	repoSourceDir="$1"
	repoIndexDir="$2"

	if test "$3" = 'check-prerequisites'; then
		for checkCommand in 'dpkg-architecture' 'dpkg-scanpackages' 'apt-ftparchive'; do
			command -v "${checkCommand}" >/dev/null 2>&1 && continue
			reportError "${checkCommand}: command not found - cannot index repositories for ${targetOs}"
			return "${RC_NOT_SUPPORTED}"
		done
		return 0
	fi

	rm -rf "${workDir}/index"
	mkdir -p "${workDir}/index"

	# In container mode, do the rest inside a container instead.
	# NB we bring in the Ubuntu indexing functions too, since the Ubuntu
	# components use the Debian ones.
	if ! ${nativeOnly}; then
		if ! test "${action}" = 'inside-container'; then
			loadComponent 'containers'
			runInContainer \
			  "volume:$(realpath "${repoSourceDir}"):/mnt/packages:ro" \
			  'containers index-repository-debian index-repository-ubuntu' \
			  'index-repository' '/mnt/packages' "${repoIndexDir}" \
			|| return $?
			index_repository_replace_debian "${repoIndexDir}" || return $?
			return 0
		fi
	fi

	# Set up the GPG keyring and $signingKey if appropriate.
	setupSigningKey || return $?

	# Subdirectory name for this type of architecture.
	binaryDir="binary-$(dpkg-architecture -q DEB_HOST_ARCH)"

	# Repository component name.
	repoComponent='main'

	# Lots of moving around with "cd" required, so use a subshell here.
	(
	set -e

	cd "${repoSourceDir}"
	dpkg-scanpackages -m . > "${workDir}/index/Packages"

	cd "${workDir}/index"

	gzip -9 < 'Packages' > 'Packages.gz'

	mkdir "${repoComponent}"
	mkdir "${repoComponent}/${binaryDir}"
	cp 'Packages' 'Packages.gz' "${repoComponent}/${binaryDir}/"

	apt-ftparchive release . > "${workDir}/Release"
	mv "${workDir}/Release" "${workDir}/index/Release"
	) || return "${RC_LOCAL_FAULT}"

	# Sign the release file if necessary.
	if test -n "${signingKey}"; then
		if test -s "${workDir}/signing-passphrase"; then
			gpg --batch --pinentry-mode loopback --passphrase-file "${workDir}/signing-passphrase" --default-key "${signingKey}" --armor --clearsign "${workDir}/index/Release" >/dev/null
		else
			gpg --default-key "${signingKey}" --armor --clearsign "${workDir}/index/Release" >/dev/null
		fi
		mv "${workDir}/index/Release.asc" "${workDir}/index/InRelease"
	fi

	# Copy the generated packages to the destination.
	if ! test "${action}" = 'inside-container'; then
		index_repository_replace_debian "${repoIndexDir}" || return $?
	fi

	return 0
}
