#!/bin/sh
#
# Core functions and main program entry point.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Adjust the default path.
# Note that /usr/local is vital on BSDs for tools like gpg.
export PATH="/sbin:/bin:/usr/sbin:/usr/bin:/usr/local/bin:/usr/local/sbin:$PATH"

# Explicitly set the umask.
umask 0022

#
# Core functions.
#

# Output the message $1 to standard error, prefixed with the program name
# and a colon.
#
reportError () {
	printf '%s: %s\n' "${PROGRAM_NAME}" "$1" >&2
}

# Call reportError and provide an additional reminder about how to check the
# program syntax.
#
reportArgumentsError () {
	reportError "$*"
	printf '%s\n' "Try \`${PROGRAM_NAME} --help' for more information." >&2
}

# Output the path of this script.
#
locateSelf () {
	if test -e "${bindir}/${PROGRAM_NAME}"; then
		printf '%s\n' "${bindir}/${PROGRAM_NAME}"
		return 0
	elif test -e "${srcdir}/${PROGRAM_NAME}"; then
		printf '%s\n' "${srcdir}/${PROGRAM_NAME}"
		return 0
	elif test -e "$0"; then
		printf '%s\n' "$0"
		return 0
	fi
	reportError 'cannot locate path to this script'
	return "${RC_LOCAL_FAULT}"
}

# Output the path of component $1.
#
locateComponent () {
	if test -e "${componentDir}/$1.sh"; then
		printf '%s\n' "${componentDir}/$1.sh"
		return 0
	elif test -e "${srcdir}/src/$1.sh"; then
		printf '%s\n' "${srcdir}/src/$1.sh"
		return 0
	fi
	reportError "$1: component not found"
	return "${RC_LOCAL_FAULT}"
}

# Load the component $1.
#
loadComponent () {
	componentPath="$(locateComponent "$1")" || return $?
	. "${componentPath}"
	return $?
}

# Load the appropriate component whose name starts with $1 for the current
# ${targetOs}, first looking for an exact match, and if there isn't one,
# falling back to a match of just the OS name with no version number.  Once
# the component is loaded, sets ${osSpecificComponent} to the name of the
# component that was loaded, and ${osSpecificFunction} to the component name
# with "-" replaced with "_".
#
loadOsSpecificComponent () {
	componentBase="$1"

	# First attempt using full targetOs.
	osSpecificComponent="${componentBase}-${targetOs}"
	osSpecificFunction="$(printf '%s\n' "${osSpecificComponent}" | tr '-' '_')"
	# Do nothing if it's already loaded.
	type "${osSpecificFunction}" >/dev/null 2>&1 && return 0
	loadComponent "${osSpecificComponent}" 2>/dev/null && return 0

	# Now try with the numbers removed from the end of targetOs.
	osSpecificComponent="$(printf '%s\n' "${osSpecificComponent}" | sed 's/[0-9]*$//')"
	osSpecificFunction="$(printf '%s\n' "${osSpecificComponent}" | tr '-' '_')"
	# Do nothing if it's already loaded.
	type "${osSpecificFunction}" >/dev/null 2>&1 && return 0

	loadComponent "${osSpecificComponent}"
	return $?
}

# Run the function $1 with the remaining parameters, unless function $1
# doesn't exist, in which case any numbers at the end of its name are
# removed and the resultant function is called instead.
#
runFunctionWithGenericFallback () {
	funcName="$1"
	shift
	type "${funcName}" >/dev/null 2>&1 || funcName="$(printf '%s\n' "${funcName}" | sed 's/[0-9]*$//')"
	"${funcName}" "$@"
	return $?
}

# Extract the source archive ${sourcePath} to the directory $1.  If the top
# level of the archive contains just one directory, that directory's
# contents are copied up one level, so for instance "$1/foo-1.2.3/bar" gets
# copied to "$1/bar".  The archive itself is also copied verbatim.
#
extractSource () {
	sourceExtractionDir="$1"

	reportProgress 'extracting the source archive'

	case "${sourcePath}" in
	*.tar.gz|*.tgz) tar xzf "${sourcePath}" -C "${sourceExtractionDir}" || return "${RC_LOCAL_FAULT}" ;;
	*.tar.bz2|*.tbz2) tar xjf "${sourcePath}" -C "${sourceExtractionDir}" || return "${RC_LOCAL_FAULT}" ;;
	*.tar.xz|*.txz) tar xJf "${sourcePath}" -C "${sourceExtractionDir}" || return "${RC_LOCAL_FAULT}" ;;
	*.zip) unzip -d "${sourceExtractionDir}" "${sourcePath}" || return "${RC_LOCAL_FAULT}" ;;
	*) reportError "${sourcePath}: unknown file extension"; return "${RC_BAD_ARGS}" ;;
	esac

	# If the archive contained just one thing at the top level, assume
	# it's a directory, and copy the directory's contents up one level.
	if find "${sourceExtractionDir}" -mindepth 1 -maxdepth 1 2>/dev/null | wc -l | grep -Fqx '1'; then
		cp -al "${sourceExtractionDir}"/*/* "${sourceExtractionDir}"/
	fi

	# Copy the archive itself as well, since some build processes need
	# the original archive, not the extracted contents.
	cp "${sourcePath}" "${sourceExtractionDir}/" || return "${RC_LOCAL_FAULT}"

	return 0
}

# If ${workDir}/signing-key exists, import it into a temporary GnuPG
# keyring, setting GNUPGHOME to a temporary directory for the purpose, and
# setting $signingKey to the ID of the signing key.
#
setupSigningKey () {
	test -s "${workDir}/signing-key" || return 0

	if ! test "${action}" = 'inside-container'; then
		GNUPGHOME="${workDir}/gnupghome"
		mkdir -p "${GNUPGHOME}"
		chmod 700 "${GNUPGHOME}"
		export GNUPGHOME
	fi

	if test -s "${workDir}/signing-passphrase"; then
		if gpg --version | sed -n 1p | grep -Fq ' 2.0.'; then
			gpg --batch --passphrase-file "${workDir}/signing-passphrase" --import "${workDir}/signing-key" || return "${RC_LOCAL_FAULT}"
		else
			gpg --batch --pinentry-mode loopback --passphrase-file "${workDir}/signing-passphrase" --import "${workDir}/signing-key" || return "${RC_LOCAL_FAULT}"
		fi
	else
		gpg --import "${workDir}/signing-key" || return "${RC_LOCAL_FAULT}"
	fi
	signingKey="$(gpg --list-keys --with-colons | awk -F : '$1=="pub" {print $5}')"

	return 0
}

# Spawn the given command under an expect script which answers any prompts
# for a passphrase with the contents of ${workDir}/signing-passphrase.
#
# If that file is not present, just runs the arguments.
#
answerPassphrasePrompts () {
	if ! test -s "${workDir}/signing-passphrase"; then
		"$@"
		return $?
	fi
	cat > "${workDir}/expect" <<EOF
set fp [open \$::env(PASSPHRASEFILE)]
gets \$fp passphrase
close \$fp
spawn -noecho {*}\${argv}
set timeout 1800
expect {
  -re ".ass *phrase:" { sleep 1; send -- \$passphrase; send "\r"; exp_continue }
}
array set e [wait]
exit \$e(0)
EOF
	PASSPHRASEFILE="${workDir}/signing-passphrase" expect -f "${workDir}/expect" "$@"
}

# Report the current status to the operator.
#
reportProgress () {
	test ${verbosity} -gt 0 && printf '[%s] %s\n' "$(date '+%Y-%m-%dT%H:%M:%S')" "$*"
}

# Run the given command, but if $silenceBuildOutput is true, redirect its
# output to a file and only write the file to the original stderr if the
# return status was nonzero.
#
possiblySilenceOutput () {
	if test "${silenceBuildOutput}" = 'false'; then
		"$@"
		return $?
	fi
	"$@" > "${workDir}/commandOutput" 2>&1
	possiblySilenceOutputStatus="$?"
	if test "${possiblySilenceOutputStatus}" -eq 0; then
		rm -f "${workDir}/commandOutput"
		return 0
	fi
	cat "${workDir}/commandOutput" >&2
	return "${possiblySilenceOutputStatus}"
}

# Check that $1 is a supported target operating system, exiting immediately
# if not.  Set targetOs to $1, and set targetOsFunctionName to $1 after
# mapping it to lower case and replacing "-" with "_".  Set buildImageName
# to ${imagePrefix}${targetOsFunctionName}, which gives the name to use for
# the container image for this OS.
#
setTargetOs () {
	targetOs="$1"
	printf '%s\n' "${supportedTargets}" | tr ' ' '\n' | grep -E . | grep -Fqx "${targetOs}" \
	|| { reportArgumentsError "${targetOs}: unknown target operating system"; exit "${RC_BAD_ARGS}"; }
	targetOsFunctionName="$(printf '%s\n' "${targetOs}" | tr '-' '_' | tr '[:upper:]' '[:lower:]')"
	buildImageName="${imagePrefix}${targetOsFunctionName}"
}

# Index a repository for target operating system $1, whose packages are in
# directory $2, and write the new index to directory $3.  Loads and calls
# the appropriate repository indexing function for the target OS.
#
indexRepository () {
	repoTargetOs="$1"
	repoSourceDir="$2"
	repoIndexDir="$3"
	repoCheckArg="$4"

	prevTargetOs="${targetOs}"
	setTargetOs "${repoTargetOs}"

	# Make sure the build image is present, if using containers.
	if ! "${nativeOnly}"; then
		loadComponent 'containers' || exit $?
		buildImage || exit $?
	fi

	if loadOsSpecificComponent 'index-repository'; then
		"${osSpecificFunction}" "${repoSourceDir}" "${repoIndexDir}" "${repoCheckArg}"
		indexResult="$?"
	else
		indexResult="${RC_LOCAL_FAULT}"
	fi

	test -n "${prevTargetOs}" && setTargetOs "${prevTargetOs}"
	return "${indexResult}"
}

# Add constraint $1 to the list of active constraints, exiting with an error
# if its syntax is incorrect.
#
addConstraint () {
	constraintSpec="$1"
	constraintKey="${constraintSpec%%=*}"
	constraintValue="${constraintSpec#*=}"
	case "${constraintKey}" in
	'')
		reportArgumentsError '--constraint: missing constraint'
		return "${RC_BAD_ARGS}"
		;;
	'pinned'|'included'|'outdated')
		case "$(printf '%s\n' "${constraintValue}" | tr '[:upper:]' '[:lower:]')" in
		'true'|'on'|'yes'|'1') constraintValue="true" ;;
		'false'|'off'|'no'|'0') constraintValue="false" ;;
		*)
			reportArgumentsError "--constraint: ${constraintKey}: ${constraintValue}: boolean value expected"
			return "${RC_BAD_ARGS}"
			;;
		esac
		;;
	'package'|'repository'|'target')
		if ! test -n "${constraintValue}"; then
			reportArgumentsError "--constraint: ${constraintKey}: missing value"
			return "${RC_BAD_ARGS}"
		fi
		;;
	*)
		reportArgumentsError "--constraint: ${constraintKey}: unknown constraint key"
		return "${RC_BAD_ARGS}"
		;;
	esac

	if test -n "${constraints}"; then
		constraints="$(printf '%s\n%s\n' "${constraints}" "${constraintKey} ${constraintValue}")"
	else
		constraints="${constraintKey} ${constraintValue}"
	fi
	return 0
}


#
# Variable initialisation and command-line option processing.
#

# Initial values for command-line option processing.
action=''
nativeOnly=false
targetOs=''
targetOsFunctionName=''
buildImageName=''
sourcePath=''
instructionsPath=''
destinationPath=''
keyFile=''
passphraseFile=''
constraints=''
logUser="${SUDO_USER}"
test -n "${logUser}" || logUser="${LOGNAME}"
verbosity=2
silenceBuildOutput=false
minArgs=0
maxArgs=0
nonOptionArgCount=0
nonOptionArg1=''
nonOptionArg2=''
nonOptionArg3=''
unquotedNonOptionArgs=''

# Parse the command-line options to apply options and select the action.
while test $# -gt 0; do
	arg="$1"
	shift

	# Check for options.
	case "${arg}" in
	'-h'|'--help'|'help') action='help' ;;
	'-V'|'--version'|'version') action='version' ;;
	'-n'|'--native') nativeOnly=true ;;
	'-q'|'--quiet') test ${verbosity} -gt 0 && verbosity=$((verbosity-1)) ;;
	'-qq') test ${verbosity} -gt 0 && verbosity=$((verbosity-1)); test ${verbosity} -gt 0 && verbosity=$((verbosity-1)) ;;
	'-v'|'--verbose') verbosity=$((1+verbosity)) ;;
	'-vv') verbosity=$((2+verbosity)) ;;
	'-t'|'--target') targetOs="$1"; shift 2>/dev/null ;;
	'--target='*) targetOs="${arg#*=}" ;;
	'-s'|'--source') sourcePath="$1"; shift 2>/dev/null ;;
	'--source='*) sourcePath="${arg#*=}" ;;
	'-i'|'--instructions') instructionsPath="$1"; shift 2>/dev/null ;;
	'--instructions='*) instructionsPath="${arg#*=}" ;;
	'-d'|'--destination') destinationPath="$1"; shift 2>/dev/null ;;
	'--destination='*) destinationPath="${arg#*=}" ;;
	'-k'|'--key') keyFile="$1"; shift 2>/dev/null ;;
	'--key='*) keyFile="${arg#*=}" ;;
	'-p'|'--passphrase') passphraseFile="$1"; shift 2>/dev/null ;;
	'--passphrase='*) passphraseFile="${arg#*=}" ;;
	'-c'|'--constraint') addConstraint "$1" || exit $?; shift 2>/dev/null ;;
	'--constraint='*) addConstraint "${arg#*=}" || exit $? ;;
	'-u'|'--user') logUser="$1"; shift 2>/dev/null ;;
	'--user='*) logUser="${arg#*=}" ;;
	'-'*) reportArgumentsError "${arg}: unknown option"; exit "${RC_BAD_ARGS}" ;;
	esac

	if test -z "${action}"; then
		# No action set - first non-option arg should be the action.
		case "${arg}" in
		'-'*) ;;
		'build-package') action="${arg}"; minArgs=0; maxArgs=0 ;;
		'build-instructions') action="${arg}"; minArgs=0; maxArgs=0 ;;
		'update-info') action="${arg}"; minArgs=0; maxArgs=0 ;;
		'index-repository') action="${arg}"; minArgs=0; maxArgs=0 ;;
		'archive-contents') action="${arg}"; minArgs=0; maxArgs=0 ;;
		'prune') action="${arg}"; minArgs=1; maxArgs=1 ;;
		'repository-contents') action="${arg}"; minArgs=0; maxArgs=0 ;;
		'select') action="${arg}"; minArgs=2; maxArgs=2 ;;
		'pin') action="${arg}"; minArgs=2; maxArgs=3 ;;
		'update-image') action="${arg}"; minArgs=0; maxArgs=0 ;;
		'install-into-image') action="${arg}"; minArgs=1; maxArgs=999 ;;
		'targets') action="${arg}"; minArgs=0; maxArgs=0 ;;
		'inside-container') action="${arg}"; minArgs=0; maxArgs=999; break ;;
		*) reportArgumentsError "${arg}: unknown action"; exit "${RC_BAD_ARGS}" ;;
		esac
	else
		# Action already set - queue up non-option args.
		case "${arg}" in
		'-'*) ;;
		*)
			nonOptionArgCount=$((1+nonOptionArgCount))
			unquotedNonOptionArgs="${unquotedNonOptionArgs} ${arg}"
			case "${nonOptionArgCount}" in
			'1') nonOptionArg1="${arg}" ;;
			'2') nonOptionArg2="${arg}" ;;
			'3') nonOptionArg3="${arg}" ;;
			esac
		;;
		esac
	fi
done

# Early exit if no action was chosen.
test -z "${action}" && { reportArgumentsError 'no action specified'; exit "${RC_BAD_ARGS}"; }

# Early exit if the number of parameters is incorrect.
test "${nonOptionArgCount}" -lt "${minArgs}" && { reportArgumentsError "${action}: insufficient parameters"; exit "${RC_BAD_ARGS}"; }
test "${nonOptionArgCount}" -gt "${maxArgs}" && { reportArgumentsError "${action}: too many parameters"; exit "${RC_BAD_ARGS}"; }

# For the build-instructions action, force a target OS.
test "${action}" = 'build-instructions' && targetOs="${instructionsTarget}"

# If a target OS was selected, validate it, and set targetOsFunctionName and
# buildImageName.
test -n "${targetOs}" && setTargetOs "${targetOs}"

# Validate that the provided paths exist.
if test -n "${sourcePath}" && ! test -e "${sourcePath}"; then
	reportArgumentsError "--source: ${sourcePath}: not found"
	exit "${RC_BAD_ARGS}"
fi
if test -n "${instructionsPath}" && ! test -e "${instructionsPath}"; then
	reportArgumentsError "--instructions: ${instructionsPath}: not found"
	exit "${RC_BAD_ARGS}"
fi
if test -n "${destinationPath}" && ! test -e "${destinationPath}"; then
	reportArgumentsError "--destination: ${destinationPath}: not found"
	exit "${RC_BAD_ARGS}"
fi
if test -n "${keyFile}" && ! test -e "${keyFile}"; then
	reportArgumentsError "--key: ${keyFile}: not found"
	exit "${RC_BAD_ARGS}"
fi
if test -n "${passphraseFile}" && ! test -e "${passphraseFile}"; then
	reportArgumentsError "--passphrase: ${passphraseFile}: not found"
	exit "${RC_BAD_ARGS}"
fi
if test -n "${passphraseFile}" && test "${nativeOnly}" = 'true' && test "${verbosity}" -gt 0; then
	reportError 'warning: --passphrase is not reliable when used with --native'
fi

# If the verbosity level is less than 2, silence the build output, but only
# if the operator isn't going to be prompted to enter a signing passphrase.
if test ${verbosity} -lt 2; then
	if test -z "${keyFile}"; then
		silenceBuildOutput=true
	elif test -n "${passphraseFile}"; then
		# Don't silence the build output in native mode here because
		# the passphrase file will be ineffective - the "expect"
		# script that would answer the passphrase prompt won't work
		# because GPG will use an agent to interact with the
		# operator, and the agent is outside our control.
		test "${nativeOnly}" = 'false' && silenceBuildOutput=true
	else
		gpgPackets="$(LC_ALL=C gpg --list-packets "${keyFile}" 2>/dev/null)"
		# Only silence the output if we saw GPG tell us the packets
		# ("pkey[...]") *and* GPG did *not* mention "protect" as in
		# "skey[2]: [v4 protected]".
		if printf '%s\n' "${gpgPackets}" | grep -Fq 'pkey[' \
		  && ! printf '%s\n' "${gpgPackets}" | grep -Fq 'protect'; \
		then
			silenceBuildOutput=true
		fi
	fi
fi

# If we aren't running inside a container, then create the working
# directory, and set up an exit trap so it is removed on exit.
if ! test "${action}" = 'inside-container'; then
	workDir="$(mktemp -d)" || { reportError 'failed to create temporary working directory'; exit "${RC_LOCAL_FAULT}"; }
	trap 'rm -rf "${workDir}"' EXIT
	# Also copy the signing key into the working directory, in case it's
	# needed.
	test -n "${keyFile}" && cp "${keyFile}" "${workDir}/signing-key"
	# Also the passphrase file, if there is one.
	test -n "${passphraseFile}" && cp "${passphraseFile}" "${workDir}/signing-passphrase"
fi

# Run the appropriate action.
exitStatus="${RC_OK}"
case "${action}" in
'help')
	loadComponent 'help' || exit $?
	showHelp
	;;

'version')
	loadComponent 'version' || exit $?
	showVersion
	;;

'build-package')
	test -n "${targetOs}" || { reportArgumentsError "${action}: no target operating system specified"; exit "${RC_BAD_ARGS}"; }
	test -n "${sourcePath}" || { reportArgumentsError "${action}: no source specified"; exit "${RC_BAD_ARGS}"; }
	test -n "${destinationPath}" || { reportArgumentsError "${action}: no destination specified"; exit "${RC_BAD_ARGS}"; }

	# Check the destination target OS matches our current one.
	destinationTarget="$(cat "${destinationPath}/.target" 2>/dev/null)"
	if test -n "${destinationTarget}" && ! test "${destinationTarget}" = "${targetOs}"; then
		reportError "${destinationPath}: destination (${destinationTarget}) does not match target operating system (${targetOs})"
		exit "${RC_BAD_ARGS}"
	fi

	# First make sure the build image is present, if using containers.
	if ! "${nativeOnly}"; then
		loadComponent 'containers' || exit $?
		possiblySilenceOutput buildImage || exit $?
	fi

	# Extract the source.
	test -e "${workDir}/build-source" && rm -rf "${workDir}/build-source"
	mkdir -p "${workDir}/build-source"
	extractSource "${workDir}/build-source" || exit $?

	# Load the build component for the target OS.
	loadOsSpecificComponent 'build-package' || exit $?
	packageBuildFunction="${osSpecificFunction}"

	# Check that the prerequisite commands are available.
	if "${nativeOnly}"; then
		reportProgress 'checking the build prerequisites'
		possiblySilenceOutput "${packageBuildFunction}" check-prerequisites || exit $?
	fi

	# Place build instructions in ${workDir}/build-instructions/.
	rm -rf "${workDir}/build-instructions"
	if test -n "${instructionsPath}"; then
		# Instructions were provided - copy them.
		if test -f "${instructionsPath}"; then
			mkdir "${workDir}/build-instructions"
			cp "${instructionsPath}" "${workDir}/build-instructions/rpm.spec"
		else
			cp -a "${instructionsPath}/" "${workDir}/build-instructions"
		fi
	else
		# Attempt to extract instructions from the source.
		loadComponent 'extract-instructions' || exit $?
		extractFunction="extract_instructions_${targetOs}"
		mkdir "${workDir}/build-instructions"
		reportProgress 'attempting to extract build instructions from the source archive'
		if type "${extractFunction}" >/dev/null 2>&1; then
			"${extractFunction}"
		else
			extractFunction="$(printf '%s\n' "${extractFunction}" | sed 's/[0-9]*$//')"
			type "${extractFunction}" >/dev/null 2>&1 && "${extractFunction}"
		fi
		# If extraction failed, the directory will be empty, so it
		# will be removable with rmdir.
		rmdir "${workDir}/build-instructions" >/dev/null 2>&1
		if ! test -d "${workDir}/build-instructions"; then
			# There were no build instructions in the source, so
			# generate some.
			loadComponent 'build-instructions' || exit $?
			mkdir "${workDir}/build-instructions"
			reportProgress 'no build instructions found - automatically generating them instead'
			possiblySilenceOutput generateBuildInstructions "${workDir}/build-instructions" || exit $?
		fi
	fi

	# Now build the package.
	reportProgress 'starting package build'
	possiblySilenceOutput "${packageBuildFunction}"
	exitStatus="$?"
	if test "${exitStatus}" -eq 0; then
		reportProgress 'package build succeeded'
	else
		reportProgress 'package build failed - exit status:' "${exitStatus}"
	fi

	# Record the target OS for this package archive.
	if test "${exitStatus}" -eq 0 && ! test -s "${destinationPath}/.target"; then
		printf '%s\n' "${targetOs}" > "${destinationPath}/.target"
	fi
	;;

'build-instructions')
	test -n "${sourcePath}" || { reportArgumentsError "${action}: no source specified"; exit "${RC_BAD_ARGS}"; }
	test -n "${destinationPath}" || { reportArgumentsError "${action}: no destination specified"; exit "${RC_BAD_ARGS}"; }
	loadComponent 'build-instructions' || exit $?
	reportProgress 'generating build instructions from source archive'
	possiblySilenceOutput generateBuildInstructions "${destinationPath}"
	exitStatus="$?"
	if test "${exitStatus}" -eq 0; then
		reportProgress 'build instructions generated successfully'
	else
		reportProgress 'build instructions generation failed - exit status:' "${exitStatus}"
	fi
	;;

'update-info')
	test -n "${sourcePath}" || { reportArgumentsError "${action}: no source specified"; exit "${RC_BAD_ARGS}"; }
	test -n "${destinationPath}" || destinationPath="${sourcePath}.txt"
	loadComponent 'update-info' || exit $?
	reportProgress 'generating package information file'
	possiblySilenceOutput updatePackageInfo "${sourcePath}" "${sourcePath}" "${destinationPath}"
	exitStatus="$?"
	if test "${exitStatus}" -eq 0; then
		reportProgress 'package information file generated successfully'
	else
		reportProgress 'package information file generation failed - exit status:' "${exitStatus}"
	fi
	;;

'index-repository')
	test -n "${sourcePath}" || { reportArgumentsError "${action}: no source specified"; exit "${RC_BAD_ARGS}"; }
	test -n "${destinationPath}" || destinationPath="${sourcePath}"

	# If no target operating system was specified, see if we can read
	# one from the target path, or the source path.
	test -z "${targetOs}" && test -s "${destinationPath}/.target" && targetOs="$(cat "${destinationPath}/.target" 2>/dev/null)"
	test -z "${targetOs}" && test -s "${sourcePath}/.target" && targetOs="$(cat "${sourcePath}/.target" 2>/dev/null)"
	test -n "${targetOs}" || { reportArgumentsError "${action}: no target operating system specified"; exit "${RC_BAD_ARGS}"; }

	# Check that the prerequisite commands are available.
	if "${nativeOnly}"; then
		reportProgress 'checking the repository indexing prerequisites'
		possiblySilenceOutput indexRepository "${targetOs}" "${sourcePath}" "${destinationPath}" check-prerequisites || exit $?
	fi

	reportProgress 'generating repository index'
	possiblySilenceOutput indexRepository "${targetOs}" "${sourcePath}" "${destinationPath}"
	exitStatus="$?"
	if test "${exitStatus}" -eq 0; then
		reportProgress 'repository index generated successfully'
	else
		reportProgress 'repository indexing failed - exit status:' "${exitStatus}"
	fi
	;;

'archive-contents')
	test -n "${sourcePath}" || { reportArgumentsError "${action}: no source specified"; exit "${RC_BAD_ARGS}"; }
	loadComponent 'archive-contents' || exit $?
	listArchiveContents
	exitStatus="$?"
	;;

'prune')
	test -n "${sourcePath}" || { reportArgumentsError "${action}: no source specified"; exit "${RC_BAD_ARGS}"; }
	pruneCount="$(printf '%s\n' "${nonOptionArg1}" | tr -dc '0-9')"
	test -n "${pruneCount}" || { reportArgumentsError "${action}: ${nonOptionArg1}: number not recognised"; exit "${RC_BAD_ARGS}"; }
	loadComponent 'prune-archive' || exit $?
	pruneArchive
	exitStatus="$?"
	;;

'repository-contents')
	test -n "${destinationPath}" || { reportArgumentsError "${action}: no destination specified"; exit "${RC_BAD_ARGS}"; }
	loadComponent 'repository-contents' || exit $?
	listRepositoryCollectionContents
	exitStatus="$?"
	;;

'select')
	test -n "${sourcePath}" || { reportArgumentsError "${action}: no source specified"; exit "${RC_BAD_ARGS}"; }
	test -n "${destinationPath}" || { reportArgumentsError "${action}: no destination specified"; exit "${RC_BAD_ARGS}"; }
	loadComponent 'select' || exit $?
	selectPackage "${nonOptionArg1}" "${nonOptionArg2}" 'false'
	exitStatus="$?"
	;;

'pin')
	test -n "${sourcePath}" || { reportArgumentsError "${action}: no source specified"; exit "${RC_BAD_ARGS}"; }
	test -n "${destinationPath}" || { reportArgumentsError "${action}: no destination specified"; exit "${RC_BAD_ARGS}"; }
	loadComponent 'select' || exit $?
	selectPackage "${nonOptionArg1}" "${nonOptionArg2}" 'true' "${nonOptionArg3}"
	exitStatus="$?"
	;;

'update-image')
	test -n "${targetOs}" || { reportArgumentsError "${action}: no target operating system specified"; exit "${RC_BAD_ARGS}"; }
	"${nativeOnly}" && { reportArgumentsError "${action}: not compatible with --native"; exit "${RC_BAD_ARGS}"; }
	loadComponent 'containers' || exit $?
	reportProgress 'updating image'
	possiblySilenceOutput updateImage
	exitStatus="$?"
	if test "${exitStatus}" -eq 0; then
		reportProgress 'image updated successfully'
	else
		reportProgress 'image update failed - exit status:' "${exitStatus}"
	fi
	;;

'install-into-image')
	test -n "${targetOs}" || { reportArgumentsError "${action}: no target operating system specified"; exit "${RC_BAD_ARGS}"; }
	"${nativeOnly}" && { reportArgumentsError "${action}: not compatible with --native"; exit "${RC_BAD_ARGS}"; }
	loadComponent 'containers' || exit $?
	reportProgress 'installing into image'
	possiblySilenceOutput installIntoImage "${unquotedNonOptionArgs}"
	exitStatus="$?"
	if test "${exitStatus}" -eq 0; then
		reportProgress 'installation into image successful'
	else
		reportProgress 'installation into image failed - exit status:' "${exitStatus}"
	fi
	;;

'targets')
	printf '%s\n' "${supportedTargets}" | tr ' ' '\n' | grep -E .
	;;

'inside-container')
	runFromInsideContainer "$@"
	;;

esac

# TODO: option to defer reindexing when calling "select" and "pin"
# TODO: action to run deferred reindexing
# TODO: action to run an interactive menu

exit "${exitStatus}"
