#!/bin/sh
#
# Functions for pruning the contents of a package archive.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Scan through the packages under the archive directory ${sourcePath}, which
# are expected to either be directly under it, or in one subdirectory per
# target operating system.  If constraints are specified in ${constraints},
# use them to filter the packages.
#
# For each package after filtering, remove versions, oldest first, until
# there are no more than ${pruneCount} versions remaining.
#
# If ${destinationPath} is provided, skip removal of any package version
# under ${sourcePath} which is also present under ${destinationPath} (which
# indicates that it is still potentially in use).
#
pruneArchive () {
	loadComponent 'archive-contents' || return $?
	packageList="$(listArchiveContents)" || return $?
	test -n "${packageList}" || return 0

	# Work through a list of distinct package names (skipping the header
	# line).
	printf '%s\n' "${packageList}" | awk -F "\t" 'FNR>1 { print $2 }' | sort -u | {
	while read -r packageName; do

		# List all of the lines for this package, sort these lines
		# in version order, newest first, and strip the first
		# ${pruneCount} lines - the remainder are candidates for
		# removal.
		printf '%s\n' "${packageList}" \
		| awk -v "name=${packageName}" \
		      -F "\t" \
		      'FNR>1 && $2==name {
		        sortOrder=""
		        fieldCount=split($3,fields,/[.-]/)
		        for (fieldNumber=1; fieldNumber<=fieldCount; fieldNumber++) {
		          fragment=sprintf("%6s", fields[fieldNumber])
			  gsub(/ /,"0",fragment)
			  sortOrder=sortOrder fragment
		        }
		        sortOrder=sprintf("%-40s",sortOrder);
		        gsub(/ /,"0",sortOrder);
		        printf "%s\t%s\n",sortOrder,$0;
		       }' \
		| sort -r \
		| sed "1,${pruneCount}d" \
		| {
		IFS="$(printf '\t')"
		while read -r sortOrder filename name version architecture os restOfLine; do
			test -n "${sortOrder}${architecture}${restOfLine}" || continue
			test -n "${filename}" || continue
			test -n "${version}" || continue
			test "${packageName}" = "${name}" || continue

			# Not a candidate for removal if it exists under
			# ${destinationPath}.
			if test -n "${destinationPath}"; then
				find "${destinationPath}"/*/"${os}" -mindepth 1 -maxdepth 1 -type f -name "${filename}" 2>/dev/null \
				| grep -Ec . \
				| grep -Fqx '0' \
				|| continue
			fi

			# Candidate for removal - remove it and its text file.
			for checkDir in "${sourcePath}" "${sourcePath}/${os}"; do
				test -d "${checkDir}/" || continue
				if test -e "${checkDir}/${filename}"; then
					printf '%s\n' "${checkDir}/${filename}"
					rm "${checkDir}/${filename}"
				fi
				test -e "${checkDir}/${filename}.txt" && rm "${checkDir}/${filename}.txt"
			done
		done
		}
	done
	}

	return 0
}
